package dao

import (
	"code.wuban.net.cn/movabridge/token-bridge/config"
	. "code.wuban.net.cn/movabridge/token-bridge/constant"
	"code.wuban.net.cn/movabridge/token-bridge/contract/bridge"
	dbModel "code.wuban.net.cn/movabridge/token-bridge/model/db"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	log "github.com/sirupsen/logrus"
	"golang.org/x/crypto/sha3"
	"math/big"
	"strings"
)

type ChainInterface interface {
	Name() string
	GetChain() *config.ChainConfig
	ParseTransferOut(log types.Log) (*bridge.BridgeContractTransferOut, error)
	ParseTransferIn(log types.Log) (*bridge.BridgeContractTransferIn, error)
	ParseTransferInExecution(log types.Log) (*bridge.BridgeContractTransferInExecution, error)
	ParseTransferInRejection(log types.Log) (*bridge.BridgeContractTransferInRejection, error)
	ParseTransferInConfirmation(log types.Log) (*bridge.BridgeContractTransferInConfirmation, error)
}

var (
	bridgeAbi, _                = bridge.BridgeContractMetaData.GetAbi()
	TransferOutEvent            = bridgeAbi.Events[EVENT_TRANSFER_OUT]
	TransferInEvent             = bridgeAbi.Events[EVENT_TRANSFER_IN]
	TransferInExecutionEvent    = bridgeAbi.Events[EVENT_TRANSFER_IN_EXECUTION]
	TransferInRejectionEvent    = bridgeAbi.Events[EVENT_TRANSFER_IN_REJECTION]
	TransferInConfirmationEvent = bridgeAbi.Events[EVENT_TRANSFER_IN_CONFIRMATION]
)

func (s *Dao) HandleEvents(chain ChainInterface, logs []types.Log) error {
	s.handleMux.Lock()
	defer s.handleMux.Unlock()

	cname := chain.Name()

	// begin orm transaction
	ormTx, err := s.BeginTx()
	if err != nil {
		log.WithField("chain", cname).WithError(err).Error("begin db transaction")
		return err
	}

	var ormTxErr error
	for _, txLog := range logs {
		if err := s.filterTransferOut(chain, txLog, ormTx); err != nil {
			ormTxErr = err
			break
		}
		if err := s.filterTransferIn(chain, txLog, ormTx); err != nil {
			ormTxErr = err
			break
		}
		if err := s.filterValidatorEvents(chain, txLog, ormTx); err != nil {
			ormTxErr = err
			break
		}
	}
	// Commit or rollback transaction based on error
	if ormTxErr != nil {
		if rbErr := ormTx.Rollback(); rbErr != nil {
			log.WithField("chain", cname).WithError(rbErr).Error("failed to rollback transaction")
		}
		log.WithField("chain", cname).WithError(ormTxErr).Error("error processing logs, transaction rolled back")
	} else {
		if cmtErr := ormTx.Commit(); cmtErr != nil {
			log.WithField("chain", cname).WithError(cmtErr).Error("failed to commit transaction")
		}
	}
	return nil
}

// filterTransferOut 用户从当前链跨出事件.
func (s *Dao) filterTransferOut(chain ChainInterface, txLog types.Log, tx *Transaction) error {
	if len(txLog.Topics) == 0 {
		return nil
	}

	if txLog.Topics[0].Hex() == TransferOutEvent.ID.Hex() {
		event, err := chain.ParseTransferOut(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferOut log")
			return err
		}
		// 防止重复入库.
		eventHash := s.transferHash(event.FromChainID.Int64(), event.OutId.Int64())
		existEvent, _ := s.GetBridgeEventByHashTx(tx, eventHash)
		if existEvent == nil {
			dbEvent := &dbModel.BridgeEvent{
				FromChain:       event.FromChainID.Int64(),
				OutTimestamp:    int64(txLog.BlockTimestamp),
				FromContract:    strings.ToLower(txLog.Address.String()),
				FromAddress:     strings.ToLower(event.Sender.String()),
				FromToken:       strings.ToLower(event.Token.String()),
				FromChainTxHash: strings.ToLower(txLog.TxHash.String()),
				SendAmount:      event.Amount.Text(10),
				FeeAmount:       event.Fee.Text(10),
				ToToken:         strings.ToLower(event.ReceiveToken.String()),
				ReceiveAmount:   new(big.Int).Sub(event.Amount, event.Fee).Text(10),
				OutId:           event.OutId.Int64(),
				Receiver:        strings.ToLower(event.Receiver.String()),
				ToChain:         event.ToChainID.Int64(),
				ToChainStatus:   TransferChainNoProcess,
				Hash:            eventHash,
			}

			err = s.CreateBridgeEventTx(tx, dbEvent)
			if err != nil {
				log.WithField("chain", chain.Name()).WithFields(log.Fields{
					"error": err.Error(),
				}).Error("db create bridge in event")
				return err
			}
			log.WithField("chain", chain.Name()).WithField("txHash", txLog.TxHash.Hex()).Info("db create, TransferOut event")
		} else {
			if existEvent.FromChainTxHash == "" {
				// update existing event with missing fields.
				existEvent.FromChain = event.FromChainID.Int64()
				existEvent.OutTimestamp = int64(txLog.BlockTimestamp)
				existEvent.FromContract = strings.ToLower(txLog.Address.String())
				existEvent.FromAddress = strings.ToLower(event.Sender.String())
				existEvent.FromToken = strings.ToLower(event.Token.String())
				existEvent.FromChainTxHash = strings.ToLower(txLog.TxHash.String())
				existEvent.SendAmount = event.Amount.Text(10)
				existEvent.FeeAmount = event.Fee.Text(10)
				existEvent.ToToken = strings.ToLower(event.ReceiveToken.String())
				existEvent.ReceiveAmount = new(big.Int).Sub(event.Amount, event.Fee).Text(10)
				existEvent.OutId = event.OutId.Int64()
				existEvent.Receiver = strings.ToLower(event.Receiver.String())
				existEvent.ToChain = event.ToChainID.Int64()
				existEvent.Hash = eventHash
				err = s.UpdateFullBridgeTx(tx, existEvent)
				if err != nil {
					log.WithField("chain", chain.Name()).WithFields(log.Fields{
						"error": err.Error(),
					}).Error("db update full bridge event")
					return err
				}
			} else {
				// already exist, do nothing.
			}
		}

	}
	return nil
}

// filterTransferIn 用户从目标链跨入事件及执行结束事件.
func (s *Dao) filterTransferIn(chain ChainInterface, txLog types.Log, tx *Transaction) error {
	if len(txLog.Topics) == 0 {
		return nil
	}

	switch txLog.Topics[0].Hex() {
	case TransferInEvent.ID.Hex():
		event, err := chain.ParseTransferIn(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferIn log")
			return err
		}
		eventHash := s.transferHash(event.FromChainID.Int64(), event.OutId.Int64())
		existEvent, _ := s.GetBridgeEventByHashTx(tx, eventHash)
		if existEvent == nil {
			dbEvent := &dbModel.BridgeEvent{
				Hash:          eventHash,
				ToContract:    strings.ToLower(txLog.Address.String()),
				InTimestamp:   int64(txLog.BlockTimestamp),
				InId:          event.InId.Int64(),
				ToChain:       chain.GetChain().ChainId,
				ToChainTxHash: strings.ToLower(txLog.TxHash.String()),
				ToChainStatus: TransferChainWaitConfirm,
			}

			err = s.CreateBridgeEventTx(tx, dbEvent)
			if err != nil {
				log.WithField("chain", chain.Name()).WithFields(log.Fields{
					"error": err.Error(),
				}).Error("db create bridge in event")
				return err
			}
			log.WithField("chain", chain.Name()).WithField("txHash", txLog.TxHash.Hex()).Info("db create, TransferOut event")
		} else {
			if existEvent.ToChainTxHash == "" {
				// update the event with the transfer in information.
				existEvent.ToContract = strings.ToLower(txLog.Address.String())
				existEvent.InTimestamp = int64(txLog.BlockTimestamp)
				existEvent.InId = event.InId.Int64()
				existEvent.ToChainTxHash = strings.ToLower(txLog.TxHash.String())
				existEvent.ToChainStatus = TransferChainWaitConfirm
				log.WithFields(log.Fields{
					"chain":       chain.Name(),
					"inTimestamp": txLog.BlockTimestamp,
				}).Debug("got transfer in event")
				if err := s.UpdateFullBridgeTx(tx, existEvent); err != nil {
					log.WithField("chain", chain.Name()).WithFields(log.Fields{
						"error": err.Error(),
					}).Error("db update transfer in event")
					return err
				}
				return nil
			} else {
				// already exist, do nothing.
				return nil
			}
		}

	case TransferInExecutionEvent.ID.Hex():
		event, err := chain.ParseTransferInExecution(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferInExecution log")
			return err
		}
		dbEvent, err := s.GetBridgeEventWithInInfoTx(tx, chain.GetChain().ChainId, event.InId.Int64())
		if err == ErrRecordNotFound {
			log.WithField("chain", chain.Name()).WithField("inId", event.InId.Int64()).Error("transfer in event not found")
			return nil
		}
		dbEvent.ToChainStatus = TransferChainExecuted
		dbEvent.FinishTxHash = strings.ToLower(txLog.TxHash.String())
		if err := s.UpdateBridgeResultTx(tx, dbEvent, dbEvent.FinishTxHash, dbEvent.ToChainStatus); err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"error": err.Error(),
			}).Error("db update transfer in execution event")
			return err
		}
	case TransferInRejectionEvent.ID.Hex():
		event, err := chain.ParseTransferInRejection(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferInExecution log")
			return err
		}
		dbEvent, err := s.GetBridgeEventWithInInfoTx(tx, chain.GetChain().ChainId, event.InId.Int64())
		if err == ErrRecordNotFound {
			log.WithField("chain", chain.Name()).WithField("inId", event.InId.Int64()).Error("transfer in event not found")
			return nil
		}
		dbEvent.ToChainStatus = TransferChainRejected
		dbEvent.FinishTxHash = strings.ToLower(txLog.TxHash.String())
		if err := s.UpdateBridgeResultTx(tx, dbEvent, dbEvent.FinishTxHash, dbEvent.ToChainStatus); err != nil {
			log.WithField("chain", chain.Name()).WithFields(log.Fields{
				"error": err.Error(),
			}).Error("db update transfer in execution event")
			return err
		}
	}
	return nil
}

// filterValidatorEvents 当前链验证者事件.
func (s *Dao) filterValidatorEvents(chain ChainInterface, txLog types.Log, tx *Transaction) error {
	if len(txLog.Topics) == 0 {
		return nil
	}
	var (
		chainId               = chain.GetChain().ChainId
		validator             = ""
		inId                  = int64(0)
		eventHash             = ""
		txHash                = txLog.TxHash.Hex()
		valOp     ValidatorOp = TransferChainNoProcess
	)
	isMyselfOp := false
	valAddr := strings.ToLower(s.GetChainValidatorAddr(chain.GetChain()).Hex())

	switch txLog.Topics[0].Hex() {
	case TransferInConfirmationEvent.ID.Hex():
		event, err := chain.ParseTransferInConfirmation(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferInConfirmation log")
			return err
		}
		valOp = ValidatorStatusConfirmation
		eventHash = validatorEventHash(chainId, event.Validator.String(), txLog.TxHash.Hex(), event.InId.Int64(), "TransferInConfirmation")
		validator = strings.ToLower(event.Validator.String())
		inId = event.InId.Int64()
		if validator == valAddr {
			isMyselfOp = true
		}
	case TransferInRejectionEvent.ID.Hex():
		event, err := chain.ParseTransferInRejection(txLog)
		if err != nil {
			log.WithField("chain", chain.Name()).WithError(err).Error("parse TransferInRejection log")
			return err
		}
		eventHash = validatorEventHash(chainId, event.Validator.String(), txLog.TxHash.Hex(), event.InId.Int64(), "TransferInRejection")
		validator = strings.ToLower(event.Validator.String())
		inId = event.InId.Int64()
		valOp = ValidatorStatusRejection
		if validator == valAddr {
			isMyselfOp = true
		}
	default:
		log.WithField("chain", chain.Name()).Error("unknown event")
		return nil
	}
	err := s.CreateValidatorEventTx(tx, txLog.Address.String(), eventHash, chainId, validator, txHash, valOp.String(), inId)
	if err != nil {
		log.WithField("chain", chain.Name()).WithFields(log.Fields{
			"error": err.Error(),
		}).Error("db create validator event")
		return err
	}
	if isMyselfOp {
		event, err := s.GetBridgeEventWithInInfoTx(tx, chainId, inId)
		if event != nil {
			if err = s.UpdateBridgeValidatorOperationTx(tx, event, int(valOp)); err != nil {
				log.WithError(err).Error("db update validator operation event")
			}
		} else {
			log.WithField("event", txLog.Topics[0].Hex()).WithError(err).Error("not found event for validator event")
		}
	}
	return nil
}

func validatorEventHash(chainId int64, validator string, txHash string, inId int64, event string) string {
	hash := sha3.NewLegacyKeccak256()
	hash.Write([]byte(fmt.Sprintf("%d%s%s%d%s", chainId, validator, txHash, inId, event)))
	return common.BytesToHash(hash.Sum(nil)).String()
}

func (s *Dao) transferHash(fromChainId int64, outId int64) string {
	chainInfo, exist := s.chainGroup[fromChainId]
	if !exist {
		return ""
	}
	fromContract := chainInfo.conf.BridgeContract
	hash := sha3.NewLegacyKeccak256()
	hash.Write([]byte(fmt.Sprintf("%d%d%s", fromChainId, outId, fromContract)))
	return common.BytesToHash(hash.Sum(nil)).String()
}

func transferOutEventHash(fromChain int64, outId int64, fromContract string) string {
	hash := sha3.NewLegacyKeccak256()
	hash.Write([]byte(fmt.Sprintf("%d%d%s", fromChain, outId, fromContract)))
	return common.BytesToHash(hash.Sum(nil)).String()
}
