package dao

import (
	"code.wuban.net.cn/movabridge/token-bridge/config"
	dbModel "code.wuban.net.cn/movabridge/token-bridge/model/db"
	"context"
	"fmt"
	"sync"
	"time"

	"github.com/ethereum/go-ethereum/ethclient"
	"gorm.io/driver/mysql"
	"gorm.io/gorm"
	"gorm.io/gorm/schema"
)

type ChainInfo struct {
	conf *config.ChainConfig
	cli  *ethclient.Client
}

type Dao struct {
	c          *config.Config
	db         *gorm.DB
	chainGroup map[int64]ChainInfo
	quit       chan struct{}
	wg         sync.WaitGroup
	handleMux  sync.Mutex
}

func New(_c *config.Config) (dao *Dao, err error) {
	dao = &Dao{
		c:          _c,
		chainGroup: make(map[int64]ChainInfo),
		quit:       make(chan struct{}),
	}

	// Connect to all configured chains
	for name, chainConfig := range _c.Chains {
		var client *ethclient.Client
		client, err = ethclient.Dial(chainConfig.RPC)
		if err != nil {
			return nil, fmt.Errorf("failed to connect to %s chain: %w", name, err)
		}

		// Get and store chain ID
		chainId, err := client.ChainID(context.Background())
		if err != nil {
			return nil, fmt.Errorf("failed to get %s chain ID: %w", name, err)
		}

		// Update the chain ID in the config
		chainConfig.ChainId = chainId.Int64()
		dao.chainGroup[chainId.Int64()] = ChainInfo{
			conf: chainConfig,
			cli:  client,
		}
		fmt.Printf("Connected to %s chain with ID %d\n", name, chainConfig.ChainId)
	}

	dsn := fmt.Sprintf("%s:%s@tcp(%s:%d)/%s?charset=utf8mb4&parseTime=True",
		_c.MySQL.User, _c.MySQL.Password, _c.MySQL.Host, _c.MySQL.Port, _c.MySQL.Database)

	// dbLogger := logger.Default.LogMode(logger.Silent)
	// if _c.Debug {
	// 	dbLogger = logger.Default.LogMode(logger.Info)
	// }

	dao.db, err = gorm.Open(mysql.Open(dsn), &gorm.Config{
		NamingStrategy: schema.NamingStrategy{
			SingularTable: true,
		},
		// Logger: dbLogger,
	})
	if err != nil {
		return
	}
	sqlDB, err := dao.db.DB()
	if err != nil {
		return
	}
	sqlDB.SetMaxOpenConns(_c.MySQL.MaxConn)
	sqlDB.SetMaxIdleConns(_c.MySQL.MaxIdleConn)
	sqlDB.SetConnMaxIdleTime(time.Hour)
	err = dao.db.AutoMigrate(&dbModel.Height{}, &dbModel.BridgeEvent{}, &dbModel.ValidatorEvent{})
	if err != nil {
		panic(err)
	}

	return dao, nil
}

func (d *Dao) Start() {
	d.wg.Add(1)
	go d.HandleTasks()
}

func (d *Dao) Stop() {
	close(d.quit)
	d.wg.Wait()

	for _, chain := range d.chainGroup {
		if chain.cli != nil {
			chain.cli.Close()
		}
	}
	if d.db != nil {
		sqlDB, _ := d.db.DB()
		sqlDB.Close()
	}
	d.chainGroup = nil
	d.db = nil
	d.c = nil
}
