package dao

import (
	"bytes"
	"code.wuban.net.cn/movabridge/token-bridge/contract/bridge"
	"encoding/hex"
	"fmt"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/ethclient"
	"golang.org/x/crypto/sha3"
	"math/big"
	"testing"
)

func TestDao_AbiEncode(t *testing.T) {
	ether := new(big.Int).Exp(big.NewInt(10), big.NewInt(18), nil)
	param := bridge.BridgesubmitParams{
		ToChainID:   big.NewInt(2),
		Receiver:    common.HexToAddress("0x000000000000000000000000000000000000dead"),
		Token:       common.HexToAddress("0x000000000000000000000000000000000000beef"),
		Amount:      new(big.Int).Mul(big.NewInt(10), ether),
		OutId:       big.NewInt(1),
		FromChainID: big.NewInt(1),
		Sender:      common.HexToAddress("0x000000000000000000000000000000000000cafe"),
		SendToken:   common.HexToAddress("0x000000000000000000000000000000000000babe"),
		SendAmount:  new(big.Int).Mul(big.NewInt(10), ether),
	}
	expectHash := common.HexToHash("0x7be55178ff6b46f92c87979ffdffc36e242f0e559556c89f1f8403a71e72e09c")
	u256Type, _ := abi.NewType("uint256", "", nil)
	addrType, _ := abi.NewType("address", "", nil)
	arguments := abi.Arguments{
		{Type: u256Type},
		{Type: u256Type},
		{Type: addrType},
		{Type: addrType},
		{Type: u256Type},
		{Type: u256Type},
		{Type: addrType},
		{Type: addrType},
	}
	data, err := arguments.Pack(param.OutId, param.FromChainID, param.Sender, param.SendToken, param.SendAmount, param.ToChainID, param.Receiver, param.Token)
	if err != nil {
		t.Fatalf("failed to encode abi: %v", err)
	}
	fmt.Println("data=0x", hex.EncodeToString(data))
	sh := sha3.NewLegacyKeccak256()
	sh.Write(data)
	signature := sh.Sum(nil)
	if bytes.Compare(signature, expectHash[:]) != 0 {
		t.Errorf("expect hash: %x, got: %x", expectHash, signature)
	}
}

func TestDao_SubmitInTransfer(t *testing.T) {
	// msg="build param" param="{269 0x3Ef35d7Db5F2d6A2fe7d16D4C8d4A20e9aD64A6A 0xF5c10392D841d55C41EBf696A4E437b2DC91f5D3 99950000000000000000 1 10323 0xfeed6dB33622Fb526a89c84A0861C29f483f1d0E 0xF5c10392D841d55C41EBf696A4E437b2DC91f5D3 100000000000000000000}" sign=26b9fea993b2e9f363fb90fdfb9ecd6a90c056abcb83af4b19239383ca1b8468
	amount, _ := new(big.Int).SetString("99950000000000000000", 10)
	samount, _ := new(big.Int).SetString("100000000000000000000", 10)
	param := bridge.BridgesubmitParams{
		ToChainID:   big.NewInt(269),
		Receiver:    common.HexToAddress("0x3Ef35d7Db5F2d6A2fe7d16D4C8d4A20e9aD64A6A"),
		Token:       common.HexToAddress("0xF5c10392D841d55C41EBf696A4E437b2DC91f5D3"),
		Amount:      amount,
		OutId:       big.NewInt(1),
		FromChainID: big.NewInt(10323),
		Sender:      common.HexToAddress("0xfeed6dB33622Fb526a89c84A0861C29f483f1d0E"),
		SendToken:   common.HexToAddress("0xF5c10392D841d55C41EBf696A4E437b2DC91f5D3"),
		SendAmount:  samount,
		Signature:   common.HexToHash("0x26b9fea993b2e9f363fb90fdfb9ecd6a90c056abcb83af4b19239383ca1b8468"),
	}
	k := "fc35cdedfab10b7218ae68b45146736bc66513452000f1fa411ff7a9c1f33439"
	//client, _ := ethclient.Dial("https://rpc.mova.bitheart.org")
	client, _ := ethclient.Dial("https://hpbnode.com")
	contract, _ := bridge.NewBridgeContract(common.HexToAddress("0x9a06d0CfAFc19a4bfe0ecd5f8bC20A26a88fA227"), client)
	signPrivateKey, err := crypto.HexToECDSA(common.Bytes2Hex(common.FromHex(k)))
	if err != nil {
		t.Fatalf("failed to convert hex to ECDSA: %v", err)
	}
	opts, err := bind.NewKeyedTransactorWithChainID(signPrivateKey, param.ToChainID)
	if err != nil {
		t.Fatalf("failed get opts: %v", err)
	}
	opts.GasLimit = 100000
	tx, err := contract.SubmitInTransfer(opts, param)
	if err != nil {
		t.Fatalf("failed to submit in transfer: %v", err)
	}
	t.Log("tx=", tx.Hash().Hex())
}
