package main

import (
	"crypto/rand"
	"encoding/hex"
	"fmt"
	"os"

	log "github.com/sirupsen/logrus"
	"github.com/spf13/cobra"
)

func generateAESKey() ([]byte, error) {
	key := make([]byte, 32) // 32 bytes = 256 bits
	_, err := rand.Read(key)
	if err != nil {
		return nil, err
	}
	return key, nil
}

var keyCmd = &cobra.Command{
	Use:   "genkey",
	Short: "Generate a random 256-bit AES key",
	Long:  "Generate a cryptographically secure random 256-bit AES key and output it in hexadecimal format",
	Run: func(cmd *cobra.Command, args []string) {
		key, err := generateAESKey()
		if err != nil {
			log.Fatal("Failed to generate AES key:", err)
		}

		keyHex := hex.EncodeToString(key)
		fmt.Printf("Generated 256-bit AES key: %s\n", keyHex)

		// Check if output file flag is provided
		outputFile, _ := cmd.Flags().GetString("output")
		if outputFile != "" {
			err := saveKeyToFile(keyHex, outputFile)
			if err != nil {
				log.Fatal("Failed to save key to file:", err)
			}
			fmt.Printf("Key saved to: %s\n", outputFile)
		}
	},
}

func saveKeyToFile(keyHex, filepath string) error {
	return os.WriteFile(filepath, []byte(keyHex), 0600)
}

func init() {
	keyCmd.Flags().StringP("output", "o", "aes.key", "File path to save the generated AES key")
	rootCmd.AddCommand(keyCmd)
}
