package dao

import (
	"code.wuban.net.cn/movabridge/token-bridge/config"
	"code.wuban.net.cn/movabridge/token-bridge/constant"
	"code.wuban.net.cn/movabridge/token-bridge/contract/bridge"
	dbModel "code.wuban.net.cn/movabridge/token-bridge/model/db"
	"context"
	"errors"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"golang.org/x/crypto/sha3"
	"math/big"
	"time"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	log "github.com/sirupsen/logrus"
)

func (d *Dao) GetBlockHeight(chain *config.ChainConfig, behindBlock ...int) (height int64, err error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	if _, ok := d.ethClient[chain.ChainId]; !ok {
		return 0, errors.New("chain client not support")
	}
	n, err := d.ethClient[chain.ChainId].BlockNumber(ctx)
	if len(behindBlock) > 0 {
		n -= uint64(behindBlock[0])
		if n < 0 {
			n = 0
		}
	}
	return int64(n), err
}

func (d *Dao) GetLatestBockHash(chain *config.ChainConfig) (hash string, err error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	if _, ok := d.ethClient[chain.ChainId]; !ok {
		return "", errors.New("chain client not support")
	}
	block, err := d.ethClient[chain.ChainId].BlockByNumber(ctx, nil)
	if err != nil {
		return
	}
	return block.Hash().Hex(), nil
}

func (d *Dao) GetBlockTime(chain *config.ChainConfig, height int) (timestamp int, err error) {
	if _, ok := d.ethClient[chain.ChainId]; !ok {
		return 0, errors.New("chain client not support")
	}
	for i := 0; i < 2; i++ {
		ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
		defer cancel()

		block, err := d.ethClient[chain.ChainId].BlockByNumber(ctx, big.NewInt(int64(height)))
		if err == nil {
			return int(block.Time()), nil
		}
	}
	return
}

func (d *Dao) GetLogs(chain *config.ChainConfig, beginHeight, endHeight int64, topics, addresses []string) (logs []types.Log, err error) {
	if _, ok := d.ethClient[chain.ChainId]; !ok {
		return nil, errors.New("chain client not support")
	}
	for i := 0; i < 2; i++ {
		// 重试2次
		logs, err = d.getLogs(chain, beginHeight, endHeight, topics, addresses)
		if err == nil {
			return logs, nil
		}
	}
	return
}

func (d *Dao) getLogs(chain *config.ChainConfig, beginHeight, endHeight int64, topics []string, addresses []string) (logs []types.Log, err error) {
	addrs := make([]common.Address, 0)
	for _, addr := range addresses {
		addrs = append(addrs, common.HexToAddress(addr))
	}

	q := ethereum.FilterQuery{
		FromBlock: big.NewInt(int64(beginHeight)),
		ToBlock:   big.NewInt(int64(endHeight)),
		Topics:    [][]common.Hash{{}},
		Addresses: addrs,
	}
	for _, topic := range topics {
		q.Topics[0] = append(q.Topics[0], common.HexToHash(topic))
	}

	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()
	return d.ethClient[chain.ChainId].FilterLogs(ctx, q)
}

func (d *Dao) buildParam(event *dbModel.BridgeEvent) (bridge.BridgesubmitParams, error) {
	sendAmount, _ := new(big.Int).SetString(event.SendAmount, 10)
	receiveAmount, _ := new(big.Int).SetString(event.ReceiveAmount, 10)
	param := bridge.BridgesubmitParams{
		ToChainID:   big.NewInt(event.ToChain),
		Receiver:    common.HexToAddress(event.Receiver),
		Token:       common.HexToAddress(event.ToToken),
		Amount:      receiveAmount,
		OutId:       big.NewInt(int64(event.OutId)),
		FromChainID: big.NewInt(event.FromChain),
		Sender:      common.HexToAddress(event.FromAddress),
		SendToken:   common.HexToAddress(event.FromToken),
		SendAmount:  sendAmount,
	}
	u256Type, _ := abi.NewType("uint256", "", nil)
	addrType, _ := abi.NewType("address", "", nil)
	arguments := abi.Arguments{
		{Type: u256Type},
		{Type: u256Type},
		{Type: addrType},
		{Type: addrType},
		{Type: u256Type},
		{Type: u256Type},
		{Type: addrType},
		{Type: addrType},
	}
	data, err := arguments.Pack(param.OutId, param.FromChainID, param.Sender,
		param.SendToken, param.SendAmount, param.ToChainID, param.Receiver, param.Token)
	if err != nil {
		return bridge.BridgesubmitParams{}, err
	}
	sh := sha3.NewLegacyKeccak256()
	sh.Write(data)
	signature := sh.Sum(nil)
	copy(param.Signature[:], signature)
	return param, nil
}

func (d *Dao) SubmitInTransfer(event *dbModel.BridgeEvent) error {

	if _, ok := d.ethClient[event.ToChain]; !ok {
		return errors.New("chain client not support")
	}
	var chain *config.ChainConfig
	for _, c := range d.c.Chains {
		if c.ChainId == event.ToChain {
			chain = c
			break
		}
	}
	if chain == nil {
		return errors.New("chain not found in config")
	}
	// verify the event is valid.
	valid := d.CheckEventValid()
	if !valid {
		log.WithField("chainId", chain.ChainId).Error("event is not valid")
		return errors.New("event is not valid")
	}

	ca, err := bridge.NewBridgeContract(common.HexToAddress(chain.BridgeContract), d.ethClient[event.ToChain])
	if err != nil {
		return err
	}
	k := chain.ValidatorPrivateKey
	signPrivateKey, err := crypto.HexToECDSA(common.Bytes2Hex(common.FromHex(k)))
	if err != nil {
		log.WithField("chainId", chain.ChainId).WithError(err).Error("failed to parse private key")
		return err
	}
	opts, err := bind.NewKeyedTransactorWithChainID(signPrivateKey, big.NewInt(int64(chain.ChainId)))
	if err != nil {
		log.WithField("chainId", chain.ChainId).WithError(err).Error("new keyed transfer failed")
		return err
	}
	param, err := d.buildParam(event)
	if err != nil {
		log.WithField("chainId", chain.ChainId).WithError(err).Error("build param failed")
		return err
	}

	if tx, err := ca.SubmitInTransfer(opts, param); err != nil {
		log.WithField("chainId", chain.ChainId).WithError(err).Error("failed to submit in transfer")
		return err
	} else {
		// update validator status.
		log.WithField("chainId", chain.ChainId).Infof("submit in transfer tx hash: %s", tx.Hash().Hex())
		return d.UpdateBridgeValidatorOperation(event, constant.ValidatorStatusConfirmation)
	}
}

func (d *Dao) CheckEventValid() bool {
	// Implement the logic to check if the event is valid.
	// This is a placeholder implementation.
	return true
}

func (d *Dao) HandleTasks() {
	defer d.wg.Done()

	ticker := time.NewTicker(5 * time.Second)
	maxCount := 5
	defer ticker.Stop()

	for {
		select {
		case <-d.quit:
			log.Info("stopping Dao task.")
			return
		case <-ticker.C:
			// select unprocessed bridge event with maxCount.
			events, err := d.GetUnprocessedBridgeEvents(maxCount)
			if err != nil {
				log.WithError(err).Error("failed to get unprocessed bridge events")
				continue
			}
			if len(events) == 0 {
				log.Info("no unprocessed bridge events found")
				continue
			}
			log.Infof("found %d unprocessed bridge events", len(events))
			for _, event := range events {
				log.WithFields(log.Fields{
					"fromChain": event.FromChain,
					"txhash":    event.FromChainTxHash,
				}).Info("processing bridge event")

				if err := d.SubmitInTransfer(event); err != nil {
					log.WithError(err).WithFields(log.Fields{
						"fromChain": event.FromChain,
						"txhash":    event.FromChainTxHash,
					}).Error("failed to submit in transfer")
				} else {
					log.WithFields(log.Fields{
						"fromChain": event.FromChain,
						"txhash":    event.FromChainTxHash,
					}).Info("successfully submitted in transfer")
				}
			}
		}
	}
}
