package main

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"
	"encoding/hex"
	"fmt"
	log "github.com/sirupsen/logrus"
	"github.com/spf13/cobra"
	"io"
	"os"
)

func encryptText(plaintext string, keyHex string) (string, error) {
	// Decode the hex key
	key, err := hex.DecodeString(keyHex)
	if err != nil {
		return "", fmt.Errorf("invalid hex key: %v", err)
	}

	// Create AES cipher
	block, err := aes.NewCipher(key)
	if err != nil {
		return "", fmt.Errorf("failed to create cipher: %v", err)
	}

	// Create GCM mode
	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return "", fmt.Errorf("failed to create GCM: %v", err)
	}

	// Generate random nonce
	nonce := make([]byte, gcm.NonceSize())
	if _, err := io.ReadFull(rand.Reader, nonce); err != nil {
		return "", fmt.Errorf("failed to generate nonce: %v", err)
	}

	// Encrypt the text
	ciphertext := gcm.Seal(nonce, nonce, []byte(plaintext), nil)
	return hex.EncodeToString(ciphertext), nil
}

var encryptCmd = &cobra.Command{
	Use:   "encrypt",
	Short: "Encrypt text using AES-256-GCM",
	Long:  "Encrypt text using AES-256-GCM encryption with the provided key",
	Run: func(cmd *cobra.Command, args []string) {
		// Get flags
		inputFile, _ := cmd.Flags().GetString("in")
		outputFile, _ := cmd.Flags().GetString("out")
		keyFile, _ := cmd.Flags().GetString("aes")

		// Read the AES key
		keyData, err := os.ReadFile(keyFile)
		if err != nil {
			log.Fatal("Failed to read key file:", err)
		}
		keyHex := string(keyData)

		// Read input text
		var plaintext string
		if inputFile != "" {
			data, err := os.ReadFile(inputFile)
			if err != nil {
				log.Fatal("Failed to read input file:", err)
			}
			plaintext = string(data)
		} else {
			log.Fatal("Input file is required")
		}

		// Encrypt the text
		encrypted, err := encryptText(plaintext, keyHex)
		if err != nil {
			log.Fatal("Failed to encrypt text:", err)
		}

		// Save to output file
		if outputFile != "" {
			err := os.WriteFile(outputFile, []byte(encrypted), 0644)
			if err != nil {
				log.Fatal("Failed to write output file:", err)
			}
			fmt.Printf("Text encrypted and saved to: %s\n", outputFile)
		} else {
			fmt.Printf("Encrypted text: %s\n", encrypted)
		}
	},
}

func init() {
	encryptCmd.Flags().StringP("in", "i", "", "Input file containing text to encrypt (required)")
	encryptCmd.Flags().StringP("out", "o", "validator.fck", "Output file for encrypted text")
	encryptCmd.Flags().StringP("aes", "k", "aes.key", "File containing the AES key")
	encryptCmd.MarkFlagRequired("in")
	rootCmd.AddCommand(encryptCmd)
}
