package chain

import (
	"code.wuban.net.cn/movabridge/token-bridge/config"
	"code.wuban.net.cn/movabridge/token-bridge/contract/bridge"
	"code.wuban.net.cn/movabridge/token-bridge/dao"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	log "github.com/sirupsen/logrus"
	"sync"
	"time"
)

var _ dao.ChainInterface = (*ChainSync)(nil)

type ChainSync struct {
	chain     *config.ChainConfig
	d         *dao.Dao
	name      string
	heightKey string
	bridgeCa  *bridge.BridgeContract
	quit      chan struct{}
	stopOnce  sync.Once
	wg        sync.WaitGroup
	syncmode  bool
}

func (s *ChainSync) ParseTransferOut(log types.Log) (*bridge.BridgeContractTransferOut, error) {
	return s.bridgeCa.ParseTransferOut(log)
}

func (s *ChainSync) ParseTransferIn(log types.Log) (*bridge.BridgeContractTransferIn, error) {
	return s.bridgeCa.ParseTransferIn(log)
}

func (s *ChainSync) ParseTransferInExecution(log types.Log) (*bridge.BridgeContractTransferInExecution, error) {
	return s.bridgeCa.ParseTransferInExecution(log)
}

func (s *ChainSync) ParseTransferInRejection(log types.Log) (*bridge.BridgeContractTransferInRejection, error) {
	return s.bridgeCa.ParseTransferInRejection(log)
}

func (s *ChainSync) ParseTransferInConfirmation(log types.Log) (*bridge.BridgeContractTransferInConfirmation, error) {
	return s.bridgeCa.ParseTransferInConfirmation(log)
}

func NewChainSync(_chain *config.ChainConfig, _d *dao.Dao) (sync *ChainSync) {
	bridgeCa, err := bridge.NewBridgeContract(common.HexToAddress(_chain.BridgeContract), nil)
	if err != nil {
		panic(err)
	}
	sync = &ChainSync{
		chain:     _chain,
		d:         _d,
		name:      _chain.Name,
		heightKey: fmt.Sprintf("%d_%s", _chain.ChainId, "height"),
		bridgeCa:  bridgeCa,
		quit:      make(chan struct{}),
		syncmode:  true,
	}

	return sync
}

func (s *ChainSync) loop() {
	defer s.wg.Done()

	finishedHeight, err := s.d.GetStorageHeight(s.heightKey)
	if err != nil {
		if err == dao.ErrRecordNotFound {
			finishedHeight = s.chain.InitialHeight
		} else {
			log.WithField("chain", s.name).WithField("chain", s.name).WithError(err).Error("get last block height")
			return
		}
	}

	var latestHeight int64
	var beginHeight = finishedHeight + 1

	log.WithField("chain", s.name).WithField("begin height", beginHeight).Info("last validator block height")

	tm := time.NewTicker(time.Second)
	defer tm.Stop()

	for {
		select {
		case <-s.quit:

			log.WithField("chain", s.name).Info("chain sync stopped")
			return
		case <-tm.C:
			var endHeight = beginHeight + int64(s.chain.BatchBlock)

			latestHeight, err = s.d.GetBlockHeight(s.chain, s.chain.BehindBlock)
			if err != nil {
				log.WithField("chain", s.name).WithError(err).Error("get latest block height")
				continue
			}

			if latestHeight <= beginHeight {
				s.syncmode = false
				continue
			}

			if beginTime, err := s.d.GetBlockTime(s.chain, beginHeight); err == nil {
				blockTime := time.Unix(int64(beginTime), 0)
				if time.Since(blockTime) < time.Minute {
					s.syncmode = false
				} else {
					s.syncmode = true
				}
				log.WithField("chain", s.name).WithFields(log.Fields{
					"begin height": beginHeight,
					"block time":   blockTime,
					"sync mode":    s.syncmode,
					"sinceTime":    time.Since(blockTime).String(),
				}).Debug("begin block time")
			}

			if latestHeight < endHeight {
				endHeight = latestHeight
			}

			if err := s.SyncLogs(beginHeight, endHeight); err != nil {
				log.WithField("chain", s.name).WithFields(log.Fields{
					"begin height": beginHeight,
					"end height":   endHeight,
				}).WithError(err).Error("sync logs failed")
				continue
			}

			if err = s.d.SetStorageHeight(s.heightKey, endHeight); err != nil {
				log.WithField("chain", s.name).WithError(err).Error("set last block height")
			}
			log.WithField("chain", s.name).WithFields(log.Fields{
				"begin height":  beginHeight,
				"end height":    endHeight,
				"latest height": latestHeight,
				"diff height":   latestHeight - endHeight,
			}).Info("validator block")

			beginHeight = endHeight + 1
			if s.syncmode {
				tm.Reset(time.Second)
			} else {
				tm.Reset(time.Second * time.Duration(s.chain.Interval))
			}
		}
	}
}

func (s *ChainSync) Name() string {
	return s.name
}

func (s *ChainSync) GetChain() *config.ChainConfig {
	return s.chain
}

func (s *ChainSync) Start() {
	s.wg.Add(1)
	go s.loop()
}

func (s *ChainSync) Stop() {
	close(s.quit)
	s.wg.Wait()
}

func (s *ChainSync) SyncLogs(beginHeight, endHeight int64) error {
	if endHeight < 0 {
		return nil
	}

	if beginHeight < 0 {
		beginHeight = 0
	}

	topics := []string{
		dao.TransferOutEvent.ID.Hex(),
		dao.TransferInEvent.ID.Hex(),
		dao.TransferInConfirmationEvent.ID.Hex(),
		dao.TransferInRejectionEvent.ID.Hex(),
		dao.TransferInExecutionEvent.ID.Hex(),
	}

	logs, err := s.d.GetLogs(s.chain, beginHeight, endHeight, topics, []string{
		s.chain.BridgeContract,
	})
	if err != nil {
		log.WithField("chain", s.name).WithFields(log.Fields{"begin": beginHeight, "end": endHeight}).WithError(err).Error("rpc: get logs")
		return err
	}

	if len(logs) > 0 {
		log.WithField("chain", s.name).WithFields(log.Fields{"begin": beginHeight, "end": endHeight}).Infof("get %d logs", len(logs))
	}

	return s.d.HandleEvents(s, logs)
}

func (s *ChainSync) IsSyncing() bool {
	return s.syncmode
}
