package dao

import (
	"code.wuban.net.cn/movabridge/token-bridge/config"
	"code.wuban.net.cn/movabridge/token-bridge/constant"
	"code.wuban.net.cn/movabridge/token-bridge/contract/bridge"
	dbModel "code.wuban.net.cn/movabridge/token-bridge/model/db"
	"context"
	"errors"
	"github.com/ethereum/go-ethereum/accounts/abi"
	"github.com/ethereum/go-ethereum/ethclient"
	"golang.org/x/crypto/sha3"
	"math/big"
	"time"

	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
	log "github.com/sirupsen/logrus"
)

func (d *Dao) GetBlockHeight(chain *config.ChainConfig, behindBlock ...int) (height int64, err error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	if _, ok := d.chainGroup[chain.ChainId]; !ok {
		return 0, errors.New("chain client not support")
	}
	chaininfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return 0, errors.New("chain client not support")
	}
	n, err := chaininfo.cli.BlockNumber(ctx)
	if len(behindBlock) > 0 {
		n -= uint64(behindBlock[0])
		if n < 0 {
			n = 0
		}
	}
	return int64(n), err
}

func (d *Dao) GetLatestBockHash(chain *config.ChainConfig) (hash string, err error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	chainInfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return "", errors.New("chain client not support")
	}
	block, err := chainInfo.cli.HeaderByNumber(ctx, nil)
	if err != nil {
		return
	}
	return block.Hash().Hex(), nil
}

func (d *Dao) GetBlockTime(chain *config.ChainConfig, height int64) (timestamp int64, err error) {
	chainInfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return 0, errors.New("chain client not support")
	}
	for i := 0; i < 2; i++ {
		ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
		defer cancel()

		block, err := chainInfo.cli.HeaderByNumber(ctx, big.NewInt(int64(height)))
		if err == nil {
			return int64(block.Time), nil
		}
	}
	return
}

func (d *Dao) GetLogs(chain *config.ChainConfig, beginHeight, endHeight int64, topics, addresses []string) (logs []types.Log, err error) {
	chainInfo, ok := d.chainGroup[chain.ChainId]
	if !ok {
		return nil, errors.New("chain client not support")
	}
	for i := 0; i < 2; i++ {
		// 重试2次
		logs, err = d.getLogs(chainInfo.cli, beginHeight, endHeight, topics, addresses)
		if err == nil {
			return logs, nil
		}
	}
	return
}

func (d *Dao) getLogs(client *ethclient.Client, beginHeight, endHeight int64, topics []string, addresses []string) (logs []types.Log, err error) {
	addrs := make([]common.Address, 0)
	for _, addr := range addresses {
		addrs = append(addrs, common.HexToAddress(addr))
	}

	q := ethereum.FilterQuery{
		FromBlock: big.NewInt(int64(beginHeight)),
		ToBlock:   big.NewInt(int64(endHeight)),
		Topics:    [][]common.Hash{{}},
		Addresses: addrs,
	}
	for _, topic := range topics {
		q.Topics[0] = append(q.Topics[0], common.HexToHash(topic))
	}

	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()
	return client.FilterLogs(ctx, q)
}

func (d *Dao) buildParam(event *dbModel.BridgeEvent) (bridge.BridgesubmitParams, error) {
	sendAmount, _ := new(big.Int).SetString(event.SendAmount, 10)
	receiveAmount, _ := new(big.Int).SetString(event.ReceiveAmount, 10)
	param := bridge.BridgesubmitParams{
		ToChainID:   big.NewInt(event.ToChain),
		Receiver:    common.HexToAddress(event.Receiver),
		Token:       common.HexToAddress(event.ToToken),
		Amount:      receiveAmount,
		OutId:       big.NewInt(int64(event.OutId)),
		FromChainID: big.NewInt(event.FromChain),
		Sender:      common.HexToAddress(event.FromAddress),
		SendToken:   common.HexToAddress(event.FromToken),
		SendAmount:  sendAmount,
	}
	u256Type, _ := abi.NewType("uint256", "", nil)
	addrType, _ := abi.NewType("address", "", nil)
	arguments := abi.Arguments{
		{Type: u256Type},
		{Type: u256Type},
		{Type: addrType},
		{Type: addrType},
		{Type: u256Type},
		{Type: u256Type},
		{Type: addrType},
		{Type: addrType},
	}
	data, err := arguments.Pack(param.OutId, param.FromChainID, param.Sender,
		param.SendToken, param.SendAmount, param.ToChainID, param.Receiver, param.Token)
	if err != nil {
		return bridge.BridgesubmitParams{}, err
	}
	sh := sha3.NewLegacyKeccak256()
	sh.Write(data)
	signature := sh.Sum(nil)
	copy(param.Signature[:], signature)

	log.WithFields(log.Fields{
		"param": param,
		"sign":  common.Bytes2Hex(param.Signature[:]),
	}).Debug("build param")
	return param, nil
}

func (d *Dao) SubmitInTransfer(event *dbModel.BridgeEvent) error {
	chain, ok := d.chainGroup[event.ToChain]
	if !ok {
		return errors.New("chain not support")
	}
	// verify the event is valid.
	valid := d.CheckEventValid()
	if !valid {
		log.WithField("chainId", chain.conf.ChainId).Error("event is not valid")
		return errors.New("event is not valid")
	}

	ca, err := bridge.NewBridgeContract(common.HexToAddress(chain.conf.BridgeContract), chain.cli)
	if err != nil {
		return err
	}

	signPrivateKey := d.validatorPk
	opts, err := bind.NewKeyedTransactorWithChainID(signPrivateKey, big.NewInt(int64(chain.conf.ChainId)))
	if err != nil {
		log.WithField("chainId", chain.conf.ChainId).WithError(err).Error("new keyed transfer failed")
		return err
	}
	opts.GasLimit = uint64(1000000)
	param, err := d.buildParam(event)
	if err != nil {
		log.WithField("chainId", chain.conf.ChainId).WithError(err).Error("build param failed")
		return err
	}

	if tx, err := ca.SubmitInTransfer(opts, param); err != nil {
		log.WithField("chainId", chain.conf.ChainId).WithError(err).Error("failed to submit in transfer")
		return d.UpdateBridgeValidatorOperation(event, constant.ValidatorStatusNoPrecess)
	} else {
		// wait tx result and update validator status.
		receipt, err := bind.WaitMined(context.Background(), chain.cli, tx)
		if err != nil {
			log.WithField("chainId", chain.conf.ChainId).WithError(err).Error("wait tx mined failed")
			return err
		}
		if receipt.Status != types.ReceiptStatusSuccessful {
			log.WithFields(log.Fields{
				"chainId": chain.conf.ChainId,
				"tx":      tx.Hash().Hex(),
			}).Error("submit in transfer receipt failed")
			//log.WithField("chainId", chain.conf.ChainId).WithField("tx", tx.Hash().Hex()).Error("tx failed")
			return d.UpdateBridgeValidatorOperation(event, constant.ValidatorStatusFailure)
		}
		// update validator status.
		log.WithFields(log.Fields{
			"tx":        tx.Hash().Hex(),
			"chainId":   chain.conf.ChainId,
			"validator": d.GetChainValidatorAddr(),
		}).Debug("submit in transfer tx succeed")
		return d.UpdateBridgeValidatorOperation(event, constant.ValidatorStatusConfirmation)
	}
}

func (d *Dao) CheckEventValid() bool {
	// Implement the logic to check if the event is valid.
	// This is a placeholder implementation.
	return true
}

func (d *Dao) HandleTasks() {
	defer d.wg.Done()

	ticker := time.NewTicker(10 * time.Second)
	maxCount := 30
	offset := 0
	defer ticker.Stop()

	for {
		select {
		case <-d.quit:
			log.Info("stopping Dao task.")
			return
		case <-ticker.C:
			if d.validatorPk == nil {
				continue
			}
			// select unprocessed bridge event with maxCount.
			events, err := d.GetUnprocessedBridgeEvents(maxCount, offset)
			if err != nil {
				log.WithError(err).Error("failed to get unprocessed bridge events")
				continue
			}
			if len(events) == 0 {
				offset = 0
				log.Info("no unprocessed bridge events found")
				continue
			}
			log.Debugf("found %d unprocessed bridge events", len(events))
			offset += len(events)

			for _, event := range events {
				//log.WithFields(log.Fields{
				//	"fromChain": event.FromChain,
				//	"txhash":    event.FromChainTxHash,
				//}).Info("processing bridge event")
				if d.IsSyncing(event.ToChain) {
					log.WithFields(log.Fields{
						"toChain": event.ToChain,
					}).Debug("target chain is syncing, skip do task")
					continue
				}

				if failed, _ := d.failedEvent.Get(event.ID); failed {
					if event.UpdatedAt.Compare(event.CreatedAt) > 0 && (int(time.Now().Sub(event.UpdatedAt).Seconds()) < d.c.TaskRetryInterval) {
						// skip recently failed tasks.
						continue
					}
				}

				if err := d.SubmitInTransfer(event); err != nil {
					log.WithError(err).WithFields(log.Fields{
						"fromChain": event.FromChain,
						"txhash":    event.FromChainTxHash,
					}).Error("failed to submit in transfer")
					d.failedEvent.Add(event.ID, true)
				} else {
					log.WithFields(log.Fields{
						"fromChain": event.FromChain,
						"txhash":    event.FromChainTxHash,
					}).Info("successfully submitted in transfer")
				}
			}
		}
	}
}

func (d *Dao) GetChainValidatorAddr() common.Address {
	k := d.validatorPk
	return crypto.PubkeyToAddress(k.PublicKey)
}
