package ipgeo

import (
	"fmt"
	"github.com/oschwald/geoip2-golang"
	"github.com/prometheus/common/log"
	"math"
	"net"
)

// EarthRadius 地球半径（单位：千米）
const EarthRadius = 6371

// Coordinates 经纬度坐标结构体
type Coordinates struct {
	Latitude  float64 // 纬度
	Longitude float64 // 经度
}

// 将角度转换为弧度
func degreesToRadians(degrees float64) float64 {
	return degrees * (math.Pi / 180.0)
}

// CalculateDistance 计算两个经纬度坐标之间的距离（单位：千米）
func calculateDistance(coord1, coord2 *Coordinates) float64 {
	lat1 := degreesToRadians(coord1.Latitude)
	lat2 := degreesToRadians(coord2.Latitude)
	lon1 := degreesToRadians(coord1.Longitude)
	lon2 := degreesToRadians(coord2.Longitude)
	// Haversine 公式
	dlon := lon2 - lon1
	dlat := lat2 - lat1
	a := math.Pow(math.Sin(dlat/2), 2) + math.Cos(lat1)*math.Cos(lat2)*math.Pow(math.Sin(dlon/2), 2)
	c := 2 * math.Atan2(math.Sqrt(a), math.Sqrt(1-a))
	distance := EarthRadius * c
	return distance
}

// GetIpAddr 获取Ip地址信息
func GetIpAddr(ipAddress string) *geoip2.City {
	// 打开 MaxMind 的 GeoIP2 数据库文件
	db, err := geoip2.Open("GeoLite2-City.mmdb")
	if err != nil {
		log.Error(" geoip2 open error:", err)
		return nil
	}
	defer func(db *geoip2.Reader) {
		err := db.Close()
		if err != nil {
			log.Error(" geoip2 close error:", err)
		}
	}(db)
	// 解析 IP 地址
	ip := net.ParseIP(ipAddress)
	// 查询 IP 地址的位置信息
	record, err := db.City(ip)
	if err != nil {
		log.Error("Get ip city error:", err)
		return nil
	}
	// 打印经纬度信息
	fmt.Printf("IP地址: %s\n", ipAddress)
	fmt.Printf("IP地址: %s\n", record.City.Names)
	fmt.Printf("经度: %f\n", record.Location.Longitude)
	fmt.Printf("纬度: %f\n", record.Location.Latitude)
	//res := &Coordinates{
	//	Latitude:  record.Location.Latitude,
	//	Longitude: record.Location.Longitude,
	//}
	return record
}
