package models

import "time"

type PayMethodType int

const (
	WeixinPay PayMethodType = iota + 1
	AliPay
	UnionPay
	PayPal
	ApplePay
	ManualPay
)

func (m PayMethodType) String() string {
	switch m {
	case WeixinPay:
		return "微信支付"
	case AliPay:
		return "支付宝"
	case UnionPay:
		return "银联支付"
	case PayPal:
		return "PayPal"
	case ApplePay:
		return "苹果支付"
	case ManualPay:
		return "手动充值"
	default:
		return "_"
	}
}

func (m PayMethodType) EnString() string {
	switch m {
	case WeixinPay:
		return "WeChatPay"
	case AliPay:
		return "AliPay"
	case UnionPay:
		return "UnionPay"
	case PayPal:
		return "PayPal"
	case ApplePay:
		return "ApplePay"
	case ManualPay:
		return "Manual Recharge"
	default:
		return "_"
	}
}

type TradeFlowType int

const (
	Income TradeFlowType = iota + 1
	Expenditure
)

func (m TradeFlowType) String() string {
	switch m {
	case Income:
		return "收入"
	case Expenditure:
		return "支出"
	default:
		return "未知支付方式"
	}
}

func (m TradeFlowType) EnString() string {
	switch m {
	case Income:
		return "Income"
	case Expenditure:
		return "Expenditure"
	default:
		return "unknown"
	}
}

type TradeKind int

const (
	Charge TradeKind = iota + 1
	Spending
	Withdrawal
)

func (m TradeKind) String() string {
	switch m {
	case Charge:
		return "充值"
	case Spending:
		return "消费"
	case Withdrawal:
		return "提现"
	default:
		return "未知支付方式"
	}
}

func (m TradeKind) EnString() string {
	switch m {
	case Charge:
		return "ReCharge"
	case Spending:
		return "Spending"
	case Withdrawal:
		return "Withdrawal"
	default:
		return "unknown"
	}
}

type PayStatus int

const (
	InitiatePay PayStatus = iota + 1
	PendingPay
	SuccessPay
	FinishCharge
	Billed
	InitiateWithdrawal
	PendingAudit
	ApprovedAudit
	RejectedAudit
	PendingTransfer
	SuccessTransfer
	SuccessWithdrawal
	FailurePay
	FailureTransfer
)

func (m PayStatus) String() string {
	switch m {
	case InitiatePay:
		return "发起充值"
	case PendingPay:
		return "待支付"
	case SuccessPay:
		return "支付成功"
	case FinishCharge:
		return "充值完成"
	case Billed:
		return "已出账"
	case InitiateWithdrawal:
		return "申请提现"
	case PendingAudit:
		return "待审核"
	case ApprovedAudit:
		return "提现申请通过"
	case RejectedAudit:
		return "提现申请被拒绝"
	case PendingTransfer:
		return "待转账"
	case SuccessTransfer:
		return "转账成功"
	case SuccessWithdrawal:
		return "转账成功"
	case FailurePay:
		return "支付失败"
	case FailureTransfer:
		return "提现失败"
	default:
		return "未知"
	}
}

func (m PayStatus) EnString() string {
	switch m {
	case InitiatePay:
		return "Initiate ReCharge"
	case PendingPay:
		return "Pending Pay"
	case SuccessPay:
		return "Success Pay"
	case FinishCharge:
		return "Success ReCharge"
	case Billed:
		return "Billed"
	case InitiateWithdrawal:
		return "Initiate Withdrawal"
	case PendingAudit:
		return "Pending Audit"
	case ApprovedAudit:
		return "Approved Audit"
	case RejectedAudit:
		return "Rejected Audit"
	case PendingTransfer:
		return "Pending Transfer"
	case SuccessTransfer:
		return "Success Transfer"
	case SuccessWithdrawal:
		return "Success Withdrawal"
	case FailurePay:
		return "Failure Pay"
	case FailureTransfer:
		return "Failure Withdrawal"
	default:
		return "unknown"
	}
}

type ChargeRequest struct {
	Amount        float64       `json:"amount"`
	PaymentMethod PayMethodType `json:"pay_method"`
}

type ChargeRecord struct {
	Id            int       `orm:"column(id);auto"`
	UserId        int       `json:"user_id";orm:"column(user_id)"`
	Amount        int64     `json:"amount";orm:"column(amount);size(7)"`
	PayMethod     int       `json:"pay_method";orm:"column(pay_method);size(1)"`
	PayMethodDesc string    `json:"pay_method_desc";orm:"column(pay_method_desc);size(20)"`
	Status        int       `json:"status";orm:"column(status);size(1)"`
	ChargeTime    time.Time `json:"charge_time";orm:"column(charge_time);type(datetime)"`
	CreatedTime   time.Time `json:"created_time";orm:"column(created_time);type(datetime)"`
	UpdatedTime   time.Time `json:"updated_time";orm:"column(updated_time);type(datetime)"`
	Deleted       int       `json:"deleted";orm:"column(deleted);size(1)"`
}

type Funds struct {
	Id            string `json:"id";orm:"column(id)"`                                // int64
	Uid           string `json:"uid";orm:"column(uid)"`                              // int
	Amount        string `json:"amount";orm:"column(amount)"`                        // int64
	TradeChannel  string `json:"trade_channel";orm:"column(trade_channel)"`          // int
	ChannelSerial string `json:"channel_serial";orm:"column(channel_serial)"`        // string
	Status        string `json:"status";orm:"column(status);size(1)"`                // int
	TradeTime     string `json:"trade_time";orm:"column(trade_time);type(datetime)"` // string
	TradeFlow     string `json:"trade_flow";orm:"column(trade_flow)"`                // int
	TradeType     string `json:"trade_type";orm:"column(trade_type)"`                // int
	Balance       string `json:"balance";orm:"column(balance)"`                      // int64
	Remark        string `json:"remark";orm:"column(remark)"`                        // string
	OrderId       string `json:"order_id";orm:"column(order_id)"`                    // string
}

type ResponseFunds struct {
	Id             string  `json:"id";orm:"column(id)"`         // int64
	Uid            string  `json:"uid";orm:"column(uid)"`       // int
	Amount         float64 `json:"amount";orm:"column(amount)"` // int64
	TradeChannel   string  `json:"trade_channel";orm:"column(trade_channel)"`
	TradeChannelEn string  `json:"trade_channel_en";orm:"column(trade_channel)"` // int
	ChannelSerial  string  `json:"channel_serial";orm:"column(channel_serial)"`  // string
	Status         int     `json:"status";orm:"column(status);size(1)"`          // int
	StatusDesc     string  `json:"status_desc,omitempty"`
	StatusDescEn   string  `json:"status_desc_en,omitempty"`
	TradeTime      string  `json:"trade_time";orm:"column(trade_time);type(datetime)"` // string
	TradeFlow      string  `json:"trade_flow";orm:"column(trade_flow)"`                // int
	TradeType      string  `json:"trade_type";orm:"column(trade_type)"`                // int
	TradeFlowEn    string  `json:"trade_flow_en";orm:"column(trade_flow)"`             // int
	TradeTypeEn    string  `json:"trade_type_en";orm:"column(trade_type)"`
	Balance        float64 `json:"balance";orm:"column(balance)"`   // int64
	Remark         string  `json:"remark";orm:"column(remark)"`     // string
	OrderId        string  `json:"order_id";orm:"column(order_id)"` // string
	UserName       string  `json:"username,omitempty"`
}

type IncomeAndExpenseRsponse struct {
	Uid    string `json:"uid";orm:"column(uid)"` // int
	Amount string `json:"amount";orm:"column(amount)"`
	Time   string `json:"time";orm:"column(time);type(datetime)"`
}
