package controllers

import (
	"ai_developer_admin/libs/mysql"
	"ai_developer_admin/libs/odysseus"
	"ai_developer_admin/models"
	"encoding/json"
	"github.com/beego/beego/v2/core/logs"
	"github.com/odysseus/cache/model"
	"net/http"
	"time"
)

type WhitelistController struct {
	MainController
}

func (server *WhitelistController) Lists() {
	token, err := server.Check()
	if err != nil {
		server.respond(http.StatusUnauthorized, err.Error())
		return
	}
	if !(token.Role == 1 || token.Role == 2) {
		server.respond(models.BusinessFailed, "Only system administrators or super administrators can perform this operation")
		return
	}
	body := server.Ctx.Input.RequestBody
	appRequest := models.AppRequest{}
	err = json.Unmarshal(body, &appRequest) //解析body中数据
	logs.Debug("appRequest", appRequest, string(body))

	if appRequest.Page == 0 {
		appRequest.Page = 1
	}
	if appRequest.Size == 0 {
		appRequest.Size = 10
	}
	offset := (appRequest.Page - 1) * appRequest.Size

	qs := mysql.GetMysqlInstace().Ormer.QueryTable("whitelist").Filter("deleted", 0)
	infoQs := qs.Offset(offset).Limit(appRequest.Size)
	count, err := infoQs.Count()
	logs.Debug("lists = ", count)
	var lists []*models.Whitelist
	if count > 0 {
		infoQs.All(&lists)
	}
	total, err := qs.Count()
	responseData := struct {
		Total int64       `json:"total"`
		Data  interface{} `json:"data,omitempty"`
	}{
		Total: total,
		Data:  lists,
	}
	server.respond(http.StatusOK, "", responseData)
}

func (server *WhitelistController) Add() {
	token, err := server.Check()
	if err != nil {
		server.respond(http.StatusUnauthorized, err.Error())
		return
	}

	if !(token.Role == 1 || token.Role == 2) {
		server.respond(models.BusinessFailed, "Only system administrators or super administrators can perform this operation")
		return
	}
	body := server.Ctx.Input.RequestBody
	logs.Debug("AddLevel body", string(body))
	request := models.Whitelist{}
	err = json.Unmarshal(body, &request) //解析body中数据
	logs.Debug("request", request)
	if err != nil {
		server.respond(models.NoRequestBody, err.Error())
		return
	}

	if request.ProfitAddress == "" {
		server.respond(models.MissingParameter, "Missing Profit Address")
		return
	}
	if request.NodeNum == 0 {
		server.respond(models.MissingParameter, "The minimum number of nodes is 1")
		return
	}

	timestamp := time.Now()

	request.CreatedTime = timestamp
	request.UpdatedTime = timestamp
	request.Deleted = 0

	_, err = mysql.GetMysqlInstace().Ormer.Insert(&request)
	if err != nil {
		server.respond(models.BusinessFailed, err.Error())
		return
	}

	server.respond(http.StatusOK, "success")
}

func (server *WhitelistController) Update() {
	token, err := server.Check()
	if err != nil {
		server.respond(http.StatusUnauthorized, err.Error())
		return
	}

	if !(token.Role == 1 || token.Role == 2) {
		server.respond(models.BusinessFailed, "Only system administrators or super administrators can perform this operation")
		return
	}
	body := server.Ctx.Input.RequestBody
	logs.Debug("AddLevel body", string(body))
	request := models.Whitelist{}
	err = json.Unmarshal(body, &request) //解析body中数据
	logs.Debug("request", request)
	if err != nil {
		server.respond(models.NoRequestBody, err.Error())
		return
	}

	if request.Id == 0 {
		server.respond(models.MissingParameter, "Missing id parameter")
		return
	}

	if request.ProfitAddress == "" {
		server.respond(models.MissingParameter, "Missing Profit Address")
		return
	}
	if request.NodeNum == 0 {
		server.respond(models.MissingParameter, "The minimum number of nodes is 1")
		return
	}
	check := models.Whitelist{Id: request.Id}
	err = mysql.GetMysqlInstace().Ormer.Read(&check)
	if err != nil {
		server.respond(models.BusinessFailed, "not found")
		return
	}

	timestamp := time.Now()

	request.CreatedTime = check.CreatedTime
	request.UpdatedTime = timestamp

	_, err = mysql.GetMysqlInstace().Ormer.Update(&request)
	if err != nil {
		server.respond(models.BusinessFailed, err.Error())
		return
	}

	whitelist := model.Whitelist{
		ID:            int64(request.Id),
		ProfitAddress: request.ProfitAddress,
		NodeNum:       int64(request.NodeNum),
		CreatedTime:   request.CreatedTime,
		UpdatedTime:   request.UpdatedTime,
		Deleted:       int64(request.Deleted),
	}

	odysseus.SetWhiteListToRedis(&whitelist)

	server.respond(http.StatusOK, "success")
}

func (server *WhitelistController) Delete() {
	token, err := server.Check()
	if err != nil {
		server.respond(http.StatusUnauthorized, err.Error())
		return
	}

	if !(token.Role == 1 || token.Role == 2) {
		server.respond(models.BusinessFailed, "Only system administrators or super administrators can perform this operation")
		return
	}
	body := server.Ctx.Input.RequestBody
	logs.Debug("AddLevel body", string(body))
	request := models.Whitelist{}
	err = json.Unmarshal(body, &request) //解析body中数据
	logs.Debug("request", request)
	if err != nil {
		server.respond(models.NoRequestBody, err.Error())
		return
	}

	if request.Id == 0 {
		server.respond(models.MissingParameter, "Missing id parameter")
		return
	}
	check := models.Whitelist{Id: request.Id}
	err = mysql.GetMysqlInstace().Ormer.Read(&check)
	if err != nil {
		server.respond(models.BusinessFailed, "not found")
		return
	}

	timestamp := time.Now()

	request.CreatedTime = check.CreatedTime
	request.UpdatedTime = timestamp
	request.Deleted = 1

	_, err = mysql.GetMysqlInstace().Ormer.Update(&request)
	if err != nil {
		server.respond(models.BusinessFailed, err.Error())
		return
	}

	whitelist := model.Whitelist{
		ID:            int64(request.Id),
		ProfitAddress: request.ProfitAddress,
		NodeNum:       int64(request.NodeNum),
		CreatedTime:   request.CreatedTime,
		UpdatedTime:   request.UpdatedTime,
		Deleted:       int64(request.Deleted),
	}

	odysseus.DeleteWhiteList(&whitelist)

	server.respond(http.StatusOK, "success")
}
