package cachedata

import (
	"github.com/odysseus/payment/model"
	"github.com/odysseus/payment/redisService"
	goredislib "github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"strconv"
	"time"
)

const (
	USER_INFO_LOCK_PRE    = "u-lock:"
	USER_INFO_KEY_PRE     = "u-info:"
	USER_INFO_DELETED_KEY = USER_INFO_KEY_PRE + "deleted:"
	USER_INFO_LEVEL_KEY   = USER_INFO_KEY_PRE + "level:"
	USER_INFO_BALANCE_KEY = USER_INFO_KEY_PRE + "bal:"
	USER_INFO_CHARGE_KEY  = USER_INFO_KEY_PRE + "charge:"

	USER_INFO_LOCK_DURATION = 10 * 1000 * 1000

	LOCK_RETRY_MAX_TIMES = 5
	LOCK_RETRY_INTERNAL  = 10 * 1000 * 1000
)

type UserInfo struct {
	Deleted int   `json:"deleted"`
	Level   int   `json:"level"`
	Balance int64 `json:"balance"`
	Charge  int64 `json:"charge"`
}

func (c *CacheData) checkExist(uid int64) (bool, error) {
	key := USER_INFO_DELETED_KEY + strconv.FormatInt(uid, 10)
	if _, err := c.rdb.Get(c.ctx, key).Result(); err == nil {
		return true, nil
	} else if err == goredislib.Nil {
		return false, nil
	} else {
		return false, err
	}
}

func (c *CacheData) getUserLock(uid int64, timeout time.Duration) (bool, func(), error) {
	lockKey := USER_INFO_LOCK_PRE + strconv.FormatInt(uid, 10)
	return redisService.TryAcquire(c.ctx, c.rdb, lockKey, timeout)
}

func (c *CacheData) getUserLockWithRetry(uid int64, timeout time.Duration) (bool, func(), error) {
	for i := 0; i < LOCK_RETRY_MAX_TIMES; i++ {
		locked, release, err := c.getUserLock(uid, timeout)
		if err != nil {
			log.WithError(err).Error("failed to acquire user lock")
			time.Sleep(LOCK_RETRY_INTERNAL)
			continue
		}
		if !locked {
			time.Sleep(LOCK_RETRY_INTERNAL)
			continue
		}
		return true, release, nil
	}
	return false, nil, nil
}

// value: 0: not deleted, 1: deleted
func (c *CacheData) UpdateUserDeleted(uid int64, value int) error {
	exist, err := c.checkExist(uid)
	if err != nil {
		return err
	}
	if !exist {
		return ErrNotExist
	}

	lockKey := USER_INFO_LOCK_PRE + strconv.FormatInt(uid, 10)
	for {
		locked, release, err := redisService.TryAcquire(c.ctx, c.rdb, lockKey, USER_INFO_LOCK_DURATION)
		if err != nil {
			log.WithError(err).Error("failed to acquire user lock")
			return err
		}
		if !locked {
			time.Sleep(USER_INFO_LOCK_DURATION)
			continue
		}
		defer release()
		key := USER_INFO_DELETED_KEY + strconv.FormatInt(uid, 10)
		err = c.rdb.Set(c.ctx, key, value, 0).Err()
		if err != nil {
			log.WithError(err).Error("failed to set user deleted")
			return err
		}

	}
	return nil
}

func (c *CacheData) UpdateUserLevel(uid int64, value int) error {
	exist, err := c.checkExist(uid)
	if err != nil {
		return err
	}
	if !exist {
		return ErrNotExist
	}
	lockKey := USER_INFO_LOCK_PRE + strconv.FormatInt(uid, 10)
	for {
		locked, release, err := redisService.TryAcquire(c.ctx, c.rdb, lockKey, USER_INFO_LOCK_DURATION)
		if err != nil {
			log.WithError(err).Error("failed to acquire user lock")
			return err
		}
		if !locked {
			time.Sleep(USER_INFO_LOCK_DURATION)
			continue
		}
		defer release()
		key := USER_INFO_LEVEL_KEY + strconv.FormatInt(uid, 10)
		err = c.rdb.Set(c.ctx, key, value, 0).Err()
		if err != nil {
			log.WithError(err).Error("failed to set user level")
			return err
		}
	}
	return nil
}

func (c *CacheData) UpdateUserBalance(uid int64, value int) error {
	exist, err := c.checkExist(uid)
	if err != nil {
		return err
	}
	if !exist {
		return ErrNotExist
	}
	lockKey := USER_INFO_LOCK_PRE + strconv.FormatInt(uid, 10)
	for {
		locked, release, err := redisService.TryAcquire(c.ctx, c.rdb, lockKey, USER_INFO_LOCK_DURATION)
		if err != nil {
			log.WithError(err).Error("failed to acquire user lock")
			return err
		}
		if !locked {
			time.Sleep(USER_INFO_LOCK_DURATION)
			continue
		}
		defer release()
		key := USER_INFO_BALANCE_KEY + strconv.FormatInt(uid, 10)
		err = c.rdb.Set(c.ctx, key, value, 0).Err()
		if err != nil {
			log.WithError(err).Error("failed to set user balance")
			return err
		}
	}
	return nil
}

// update equal set
func (c *CacheData) UpdateUserCharge(uid int64, value int) error {
	exist, err := c.checkExist(uid)
	if err != nil {
		return err
	}
	if !exist {
		return ErrNotExist
	}
	lockKey := USER_INFO_LOCK_PRE + strconv.FormatInt(uid, 10)
	for {
		locked, release, err := redisService.TryAcquire(c.ctx, c.rdb, lockKey, USER_INFO_LOCK_DURATION)
		if err != nil {
			log.WithError(err).Error("failed to acquire user lock")
			return err
		}
		if !locked {
			time.Sleep(USER_INFO_LOCK_DURATION)
			continue
		}
		defer release()
		key := USER_INFO_CHARGE_KEY + strconv.FormatInt(uid, 10)
		err = c.rdb.Set(c.ctx, key, value, 0).Err()
		if err != nil {
			log.WithError(err).Error("failed to set user charge")
			return err
		}
	}
	return nil
}

func (c *CacheData) SetUserInfo(user *model.User) error {
	info := UserInfo{
		Deleted: int(user.Deleted),
		Level:   int(user.Level),
		Balance: user.Balance,
		Charge:  0,
	}
	idStr := strconv.FormatInt(user.ID, 10)
	// set every info with pipeline
	pip := c.rdb.TxPipeline()
	// set user deleted
	pip.Set(c.ctx, USER_INFO_DELETED_KEY+idStr, info.Deleted, 0)
	// set user level
	pip.Set(c.ctx, USER_INFO_LEVEL_KEY+idStr, info.Level, 0)
	// set user balance
	pip.Set(c.ctx, USER_INFO_BALANCE_KEY+idStr, info.Balance, 0)
	// set user charge
	pip.Set(c.ctx, USER_INFO_CHARGE_KEY+idStr, info.Charge, 0)
	_, err := pip.Exec(c.ctx)
	return err
}

func (c *CacheData) GetUserInfo(uid int64) (*UserInfo, error) {
	idStr := strconv.FormatInt(uid, 10)
	// get every info with pipeline
	pip := c.rdb.Pipeline()
	// get user deleted
	deletedCmd := pip.Get(c.ctx, USER_INFO_DELETED_KEY+idStr)
	// get user level
	levelCmd := pip.Get(c.ctx, USER_INFO_LEVEL_KEY+idStr)
	// get user balance
	balCmd := pip.Get(c.ctx, USER_INFO_BALANCE_KEY+idStr)
	// get user charge
	chargeCmd := pip.Get(c.ctx, USER_INFO_CHARGE_KEY+idStr)

	_, err := pip.Exec(c.ctx)
	if err != nil {
		log.WithError(err).Error("failed to get user info")
		return nil, err
	}
	var info = new(UserInfo)

	if deletedCmd.Err() == nil && levelCmd.Err() == nil && balCmd.Err() == nil && chargeCmd.Err() == nil {
		info.Deleted, _ = strconv.Atoi(deletedCmd.Val())
		info.Level, _ = strconv.Atoi(levelCmd.Val())
		info.Balance, _ = strconv.ParseInt(balCmd.Val(), 10, 64)
		info.Charge, _ = strconv.ParseInt(chargeCmd.Val(), 10, 64)
		return info, nil
	}

	if deletedCmd.Err() == goredislib.Nil || levelCmd.Err() == goredislib.Nil || balCmd.Err() == goredislib.Nil || chargeCmd.Err() == goredislib.Nil {
		// get user from db.
		if user, err := c.userRepo.GetById(uid); err == nil {
			info.Deleted = int(user.Deleted)
			info.Level = int(user.Level)
			info.Balance = user.Balance
			info.Charge = 0

			c.SetUserInfo(user)
			return info, nil
		}
	}
	return nil, err
}
