package cachedata

import (
	"encoding/json"
	"github.com/odysseus/payment/model"
	"strconv"
)

func (c *CacheData) GetTaskWithPath(path string) (*model.TaskType, error) {
	// get from local cache
	if v, exist := c.pathLru.Get(path); exist {
		return v.(*model.TaskType), nil
	}
	// get id from redis and then get task from redis by id.
	pathIdKey := "path-id:" + path
	if taskIdKey, err := c.getTaskIdFromRedis(pathIdKey); err == nil {
		// get task from redis by id
		if task, err := c.getTaskFromRedis(taskIdKey); err == nil {
			// set task to local cache
			c.pathLru.Add(path, task)
			return task, nil
		}
	}
	// get task from db by path and then set task to redis.
	if task, err := c.getTaskFromDbByPath(path); err == nil {
		// set task to redis
		taskIdKey := "task-id-" + strconv.FormatInt(task.ID, 10)
		if data, err := json.Marshal(task); err == nil {
			if err = c.rdb.Set(c.ctx, taskIdKey, string(data), 0).Err(); err == nil {
				// set taskIdKey to redis
				if err = c.rdb.Set(c.ctx, pathIdKey, taskIdKey, 0).Err(); err == nil {
					// set task to local cache
					c.pathLru.Add(path, task)
					return task, nil
				} else {
					return nil, err
				}
			} else {
				return nil, err
			}
		} else {
			return nil, err
		}
	} else {
		return nil, err
	}
}

func (c *CacheData) getTaskFromRedis(taskIdKey string) (*model.TaskType, error) {
	if data, err := c.rdb.Get(c.ctx, taskIdKey).Result(); err == nil {
		var task = new(model.TaskType)
		if err = json.Unmarshal([]byte(data), task); err == nil {
			return task, nil
		}
		return nil, err
	} else {
		return nil, err
	}
}

func (c *CacheData) getTaskIdFromRedis(key string) (string, error) {
	// get from redis
	taskIdKey, err := c.rdb.Get(c.ctx, key).Result()
	return taskIdKey, err
}

func (c *CacheData) getTaskFromDbByPath(path string) (*model.TaskType, error) {
	return c.taskType.GetByApiPath(path)
}

func (c *CacheData) getTaskFromDbById(id int64) (*model.TaskType, error) {
	return c.taskType.GetById(id)
}

func (c *CacheData) GetTaskWithId(id int64) (*model.TaskType, error) {
	// get from redis
	taskIdKey := "task-id-" + strconv.FormatInt(id, 10)
	// get task from redis by id
	if task, err := c.getTaskFromRedis(taskIdKey); err == nil {
		return task, nil
	}
	// get task from db by id and then set task to redis.
	if task, err := c.getTaskFromDbById(id); err == nil {
		// set task to redis
		if data, err := json.Marshal(task); err == nil {
			if err = c.rdb.Set(c.ctx, taskIdKey, string(data), 0).Err(); err == nil {
				return task, nil
			} else {
				return nil, err
			}
		} else {
			return nil, err
		}
	} else {
		return nil, err
	}
}

func (c *CacheData) SetTaskDataToRedis(task *model.TaskType) error {
	taskIdKey := "task-id-" + strconv.FormatInt(task.ID, 10)
	if data, err := json.Marshal(task); err == nil {
		if err = c.rdb.Set(c.ctx, taskIdKey, string(data), 0).Err(); err == nil {
			return nil
		}
		return err
	} else {
		return err
	}
}
