package redisService

import (
	"context"
	"fmt"
	"github.com/gomodule/redigo/redis"
	"github.com/google/uuid"
	goredislib "github.com/redis/go-redis/v9"
	"strconv"
	"strings"
	"time"
)

func TryAcquire(ctx context.Context, rs *goredislib.Client, lockKey string, lockTimeout time.Duration) (acquired bool, release func(), _ error) {
	timeout := time.Now().Add(lockTimeout).UnixNano()
	lockToken := fmt.Sprintf("%d,%s", timeout, uuid.New().String())

	release = func() {
		// Best effort to check we're releasing the lock we think we have. Note that it
		// is still technically possible the lock token has changed between the GET and
		// DEL since these are two separate operations, i.e. when the current lock happen
		// to be expired at this very moment.
		get, _ := rs.Get(ctx, lockKey).Result()
		if get == lockToken {
			_ = rs.Del(ctx, lockKey)
		}
	}
	set, err := rs.SetNX(ctx, lockKey, lockToken, lockTimeout).Result()
	if err != nil {
		return false, nil, err
	} else if set {
		return true, release, nil
	}

	// We didn't get the lock, but we can check if the lock is expired.
	currentLockToken, err := rs.Get(ctx, lockKey).Result()
	if err == redis.ErrNil {
		// Someone else got the lock and released it already.
		return false, nil, nil
	} else if err != nil {
		return false, nil, err
	}

	currentTimeout, _ := strconv.ParseInt(strings.SplitN(currentLockToken, ",", 2)[0], 10, 64)
	if currentTimeout > time.Now().UnixNano() {
		// The lock is still valid.
		return false, nil, nil
	}

	// The lock has expired, try to acquire it.
	get, err := rs.GetSet(ctx, lockKey, lockToken).Result()
	if err != nil {
		return false, nil, err
	} else if get != currentLockToken {
		// Someone else got the lock
		return false, nil, nil
	}

	// We got the lock.
	return true, release, nil
}
