// SPDX-License-Identifier: GPL-3.0
pragma solidity ^0.8.20;

import {Ownable} from "@openzeppelin/contracts/access/Ownable.sol";
import {EnumerableSet} from "@openzeppelin/contracts/utils/structs/EnumerableSet.sol";

import {IAddressStorage} from "./interface/IAddressStorage.sol";

contract AddressStorage is IAddressStorage, Ownable {
    using EnumerableSet for EnumerableSet.AddressSet;

    EnumerableSet.AddressSet private containerAddressSet;

    EnumerableSet.AddressSet private nmAddressSet;

    event NmAddressChanged(address indexed addr, bool indexed ok, uint256 indexed count);

    event ContainerAddressChanged(address indexed addr, bool indexed ok, uint256 indexed count);

    constructor() Ownable(msg.sender) {
        setContainerAddress(msg.sender, true);
        setNmAddress(msg.sender, true);
    }

    function setNmAddress(address addr, bool ok) public onlyOwner {
        if (ok) {
            _addNmAddress(addr);
        } else {
            _delNmAddress(addr);
        }
        emit NmAddressChanged(addr, ok, nmAddressSet.length());
    }

    function setContainerAddress(address addr, bool ok) public onlyOwner {
        if (ok) {
            _addContainerAddress(addr);
        } else {
            _delContainerAddress(addr);
        }
        emit ContainerAddressChanged(addr, ok, containerAddressSet.length());
    }

    function isNmAddress(address addr) public view returns (bool) {
        return nmAddressSet.contains(addr);
    }

    function isContainerAddress(address addr) public view returns (bool) {
        return containerAddressSet.contains(addr);
    }

    function getNmAddresses() public view returns (address[] memory) {
        return nmAddressSet.values();
    }

    function getContainerAddresses() public view returns (address[] memory) {
        return containerAddressSet.values();
    }

    function _addNmAddress(address addr) internal {
        bool exist = nmAddressSet.contains(addr);
        require(!exist, "Already exist");
        nmAddressSet.add(addr);
    }

    function _delNmAddress(address addr) internal {
        bool exist = nmAddressSet.contains(addr);
        require(exist, "Not exist");
        nmAddressSet.remove(addr);
    }

    function _addContainerAddress(address addr) internal {
        bool exist = containerAddressSet.contains(addr);
        require(!exist, "Already exist");
        containerAddressSet.add(addr);
    }

    function _delContainerAddress(address addr) internal {
        bool exist = containerAddressSet.contains(addr);
        require(exist, "Not exist");
        containerAddressSet.remove(addr);
    }
}