// SPDX-License-Identifier: GPL-3.0
pragma solidity ^0.8.20;

import {Ownable} from "@openzeppelin/contracts/access/Ownable.sol";
import {Pausable} from "@openzeppelin/contracts/utils/Pausable.sol";
import {ReentrancyGuard} from "@openzeppelin/contracts/utils/ReentrancyGuard.sol";
import {MerkleProof} from "@openzeppelin/contracts/utils/cryptography/MerkleProof.sol";

import {IValidator} from "./interface/IValidator.sol";
import {IDistribution} from "./interface/IDistribution.sol";

contract Distribution is IDistribution, Ownable, Pausable, ReentrancyGuard {
    IValidator public validator;

    mapping(address => uint256) public userClaimed;

    event Claimed(address indexed addr, uint256 indexed amount, uint256 indexed totalClaimedAmount);

    constructor(address _validator) Ownable(msg.sender) {
        validator = IValidator(_validator);
    }

    function setPause() public onlyOwner {
        _pause();
    }

    function setUnPause() public onlyOwner {
        _unpause();
    }

    function withdraw(address addr, uint256 amount) public onlyOwner {
        transferToken(addr, amount);
    }

    function claim(bytes32[] calldata proof, uint256 userAllAmount) public whenNotPaused nonReentrant {
        bytes32 leaf = keccak256(abi.encodePacked(msg.sender, userAllAmount));
        require(MerkleProof.verifyCalldata(proof, _merkleRoot(), leaf), "Invalid proof");

        uint256 userCanClaimAmount = userAllAmount - userClaimed[msg.sender];
        userClaimed[msg.sender] = userAllAmount;
        transferToken(msg.sender, userCanClaimAmount);
        emit Claimed(msg.sender, userCanClaimAmount, userAllAmount);
    }

    function transferToken(address addr, uint256 amount) internal {
        (bool success, ) = addr.call{value: amount}("");
        require(success, "Transfer failed");
    }

    function _merkleRoot() internal returns (bytes32){
        return validator.getMerkleRoot();
    }

    function claimedOf() public view returns (uint256){
        return userClaimed[msg.sender];
    }
}