// SPDX-License-Identifier: GPL-3.0
pragma solidity ^0.8.20;

import {Ownable} from "@openzeppelin/contracts/access/Ownable.sol";
import {EnumerableSet} from "@openzeppelin/contracts/utils/structs/EnumerableSet.sol";

import {IValidator} from "./interface/IValidator.sol";

contract Validator is IValidator, Ownable {
    using EnumerableSet for EnumerableSet.AddressSet;

    EnumerableSet.AddressSet private validatorSet;

    bytes32 public merkleRoot; // 当前最新的 merkle root
    bytes32 public merkleSumRoot; // 当前最新的 merkle sum root
    uint256 public merkleRootThreshold ; // 多节点提交时，生效阈值
    uint256 public dailyMerkleRootSubmitCount; // 当天已提交次数
    uint256 public lastMerkleRootUpdate; // 上次 merkle root 更新时间

    // 记录每天每个 merkle root 提交次数 2023-01-01 -> hash(merkleRoot, merkleSumRoot) -> 1
    mapping(uint256 => mapping(bytes32 => uint256)) public dailyMerkleRootSubmitCountMap;
    // 记录每天每个 validator 是否已提交过
    mapping(uint256 => mapping(address => bool)) public dailyMerkleRootSubmittedMap;
    // 记录每天已验证的 merkle root （已超过生效阈值）
    mapping(uint256 => bool) public dailyMerkleRootVerifiedMap;
    // 记录每天的 merkle root
    mapping(uint256 => bytes32) public dailyMerkleRootMap;
    // 记录每天的 merkle sum root
    mapping(uint256 => bytes32) public dailyMerkleSumRootMap;
    // 记录每天的工作量总数
    mapping(uint256 => uint256) public dailyWorkloadMap;
    // 记录每天的奖励金额
    mapping(uint256 => uint256) public dailyRewardMap;
    // 每天奖励金额
    uint256 public dailyReward;

    constructor() Ownable(msg.sender) {
        merkleRootThreshold = 1;
        dailyReward = 1 ether;
        validatorSet.add(msg.sender);
    }

    modifier onlyValidator() {
        require(validatorSet.contains(msg.sender), "Only validator");
        _;
    }

    event MerkleRootChanged(uint256 date, bytes32 merkleRoot, bytes32 merkleSumRoot, uint256 totalWorkload);
    event MerkleRootThresholdChanged(uint256 merkleRootThreshold);
    event MerkleRootSubmitted(address indexed addr, bytes32 merkleRoot, bytes32 merkleSumRoot, uint256 indexed count, uint256 indexed dailySubmitCount);
    event DailyRewardChanged(uint256 dailyReward);
    event ValidatorChanged(address indexed addr, bool indexed ok, uint256 indexed count);

    function getMerkleRoot() public view returns (bytes32){
        return merkleRoot;
    }

    function getWorkloadReward(uint256 dailyWorkload) view public returns (uint256) {
        if (dailyWorkload == 0) {
            return 0;
        }
        return dailyReward / dailyWorkload;
    }

    function submitMerkleRoot(uint256 _date, bytes32 _merkleRoot, bytes32 _merkleSumRoot, uint256 _totalWorkload) public onlyValidator {
//        require(dailyMerkleRootSubmitCount <= validatorSet.length(), "Too many submissions");
//        require(!dailyMerkleRootSubmittedMap[_date][msg.sender], "Already submitted");
        // check already verified
        if (dailyMerkleRootVerifiedMap[_date]) {
            return;
        }
//        require(block.timestamp - lastMerkleRootUpdate >= (3600 * 24 - 300), "Too soon");
        bytes32 msgHash = keccak256(abi.encodePacked(_merkleRoot, _merkleSumRoot, _totalWorkload));

        dailyMerkleRootSubmitCountMap[_date][msgHash] ++;
        dailyMerkleRootSubmittedMap[_date][msg.sender] = true;
        dailyMerkleRootSubmitCount ++;

        if (dailyMerkleRootSubmitCountMap[_date][msgHash] >= merkleRootThreshold) {
            dailyMerkleRootVerifiedMap[_date] = true;
            _setMerkleRoot(_date, _merkleRoot, _merkleSumRoot);
            dailyWorkloadMap[_date] = _totalWorkload;
            dailyRewardMap[_date] = dailyReward;
            lastMerkleRootUpdate = block.timestamp;
            dailyMerkleRootSubmitCount = 0;
            emit MerkleRootChanged(_date, _merkleRoot, _merkleSumRoot, _totalWorkload);
        }
        emit MerkleRootSubmitted(msg.sender, _merkleRoot, _merkleSumRoot, dailyMerkleRootSubmitCountMap[_date][msgHash], dailyMerkleRootSubmitCount);
    }

    function _setMerkleRoot(uint256 _date, bytes32 _merkleRoot, bytes32 _merkleSumRoot) internal {
        merkleRoot = _merkleRoot;
        merkleSumRoot = merkleSumRoot;
        dailyMerkleRootMap[_date] = _merkleRoot;
        dailyMerkleSumRootMap[_date] = _merkleSumRoot;
    }

    function setDailyReward(uint256 _dailyReward) public onlyOwner {
        dailyReward = _dailyReward;
        emit DailyRewardChanged(_dailyReward);
    }

    function getDailyReward() public view returns (uint256) {
        return dailyReward;
    }

    function setRootThreshold(uint256 _merkleRootThreshold) public onlyOwner {
        merkleRootThreshold = _merkleRootThreshold;
        emit MerkleRootThresholdChanged(_merkleRootThreshold);
    }

    function setValidator(address addr, bool ok) public onlyOwner {
        if (ok) {
            _addValidator(addr);
        } else {
            _delValidator(addr);
        }
        emit ValidatorChanged(addr, ok, validatorSet.length());
    }

    function _addValidator(address addr) internal {
        bool exist = validatorSet.contains(addr);
        require(!exist, "Already exist");
        validatorSet.add(addr);
    }

    function _delValidator(address addr) internal {
        bool exist = validatorSet.contains(addr);
        require(exist, "Not exist");
        validatorSet.remove(addr);
    }

    function getValidators() public view returns (address[] memory) {
        return validatorSet.values();
    }

    function isValidator(address addr) public view returns (bool) {
        return validatorSet.contains(addr);
    }

    function getWorkloadByDate(uint256 _date) public view returns (uint256) {
        return dailyWorkloadMap[_date];
    }

    function getRewardByDate(uint256 _date) public view returns (uint256) {
        return dailyRewardMap[_date];
    }

}