package operator

import (
	"context"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

type WorkerInstalledInfo struct {
	ID       string `bson:"_id,omitempty" json:"id"`
	WorkerId string `bson:"worker_id" json:"worker_id"` // use worker MinerPubkey
	ModelId  int    `bson:"model_id" json:"model_id"`   // installed model.
	GpuFree  int64  `bson:"gpu_free" json:"gpu_free"`
}

type WorkerInstalledOperator struct {
	client *mongo.Client
	col    *mongo.Collection
}

func NewDBWorkerInstalled(client *mongo.Client, database string) *WorkerInstalledOperator {
	return &WorkerInstalledOperator{
		client: client,
		col:    client.Database(database).Collection("worker_running_info"),
	}
}

func (d *WorkerInstalledOperator) Insert(ctx context.Context, worker *WorkerRunningInfo) (*mongo.InsertOneResult, error) {
	return d.col.InsertOne(ctx, worker)
}

func (d *WorkerInstalledOperator) UpdateGpuFree(ctx context.Context, id string, gpuFree int64) error {
	update := bson.M{"$set": bson.M{"gpu_free": gpuFree}}
	_, err := d.col.UpdateOne(ctx, bson.M{"_id": id}, update)
	return err
}

func (d *WorkerInstalledOperator) FindWorkerByModelId(ctx context.Context, modelId int, limit int) ([]*WorkerInstalledInfo, error) {
	// find all worker that at least one installed model's mode_id is equal modelId
	// sort by wait time
	findOptions := options.Find()
	findOptions.SetLimit(int64(limit))
	findOptions.SetSort(bson.D{{"exec_time", 1}})

	selector := bson.M{"model_id": modelId}
	cursor, err := d.col.Find(ctx, selector, findOptions)
	if err != nil {
		return nil, err
	}
	defer cursor.Close(ctx)

	var workers []*WorkerInstalledInfo
	if err = cursor.All(ctx, &workers); err != nil {
		return nil, err
	}
	return workers, nil
}
