package operator

import (
	"context"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

type WorkerRunningInfo struct {
	ID       string `bson:"_id,omitempty" json:"id"`
	WorkerId string `bson:"worker_id" json:"worker_id"` // use worker MinerPubkey
	ModelId  int    `bson:"model_id" json:"model_id"`
	ExecTime int    `bson:"exec_time" json:"exec_time"`
}

type WorkerRunningOperator struct {
	client *mongo.Client
	col    *mongo.Collection
}

func NewDBWorkerRunning(client *mongo.Client, database string) *WorkerRunningOperator {
	return &WorkerRunningOperator{
		client: client,
		col:    client.Database(database).Collection("worker_running_info"),
	}
}

func (d *WorkerRunningOperator) Insert(ctx context.Context, worker *WorkerRunningInfo) (*mongo.InsertOneResult, error) {
	return d.col.InsertOne(ctx, worker)
}

func (d *WorkerRunningOperator) UpdateExecTime(ctx context.Context, id string, execTime int) error {
	update := bson.M{"$set": bson.M{"exec_time": execTime}}
	_, err := d.col.UpdateOne(ctx, bson.M{"_id": id}, update)
	return err
}

func (d *WorkerRunningOperator) FindWorkerByModelId(ctx context.Context, modelId int, limit int) ([]*WorkerRunningInfo, error) {
	// find all worker that at least one running model's mode_id is equal modelId
	// sort by wait time
	findOptions := options.Find()
	findOptions.SetLimit(int64(limit))
	findOptions.SetSort(bson.D{{"exec_time", 1}})

	selector := bson.M{"model_id": modelId}
	cursor, err := d.col.Find(ctx, selector, findOptions)
	if err != nil {
		return nil, err
	}
	defer cursor.Close(ctx)

	var workers []*WorkerRunningInfo
	if err = cursor.All(ctx, &workers); err != nil {
		return nil, err
	}
	return workers, nil
}
