package operator

import (
	"context"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"mogo/types"
)

type WorkerInfo struct {
	ID       string              `bson:"_id,omitempty" json:"id"`
	WorkerId string              `bson:"worker_id" json:"worker_id"` // use worker MinerPubkey
	NodeInfo *types.NodeInfo     `bson:"node_info" json:"node_info"`
	Models   *types.ModelInfo    `bson:"model_infos" json:"model_infos"`
	Hardware *types.HardwareInfo `bson:"hardware" json:"hardware"`
}

type WorkerInfoOperator struct {
	client *mongo.Client
	col    *mongo.Collection
}

func NewDBWorker(client *mongo.Client, database string) *WorkerInfoOperator {
	return &WorkerInfoOperator{
		client: client,
		col:    client.Database(database).Collection("workerinfo"),
	}
}

func (d *WorkerInfoOperator) InsertWorker(ctx context.Context, worker *WorkerInfo) (*mongo.InsertOneResult, error) {
	return d.col.InsertOne(ctx, worker)
}

func (d *WorkerInfoOperator) UpdateModel(ctx context.Context, id string, models *types.ModelInfo) error {
	update := bson.M{"$set": bson.M{"model_infos": models}}
	_, err := d.col.UpdateOne(ctx, bson.M{"_id": id}, update)
	return err
}

func (d *WorkerInfoOperator) UpdateHardware(ctx context.Context, id string, hardware *types.HardwareInfo) error {
	update := bson.M{"$set": bson.M{"hardware": hardware}}
	_, err := d.col.UpdateOne(ctx, bson.M{"_id": id}, update)
	return err
}

func (d *WorkerInfoOperator) UpdateNodeInfo(ctx context.Context, id string, nodeInfo *types.NodeInfo) error {
	update := bson.M{"$set": bson.M{"node_info": nodeInfo}}
	_, err := d.col.UpdateOne(ctx, bson.M{"_id": id}, update)
	return err
}

func (d *WorkerInfoOperator) FindWorkerByRunningModelAndSortByWaitTime(ctx context.Context, modelId string, limit int) ([]*WorkerInfo, error) {
	// find all worker that at least one running model's mode_id is equal modelId
	// sort by wait time
	findOptions := options.Find()
	findOptions.SetLimit(int64(limit))
	findOptions.SetSort(bson.D{{"model_infos.running_models.wait_time", 1}})

	selector := bson.M{"model_infos.running_models.model_id": modelId}
	cursor, err := d.col.Find(ctx, selector, findOptions)
	if err != nil {
		return nil, err
	}
	defer cursor.Close(ctx)

	var workers []*WorkerInfo
	if err = cursor.All(ctx, &workers); err != nil {
		return nil, err
	}
	return workers, nil
}

func (d *WorkerInfoOperator) FindWorkerByInstallModelAndSortByGpuRam(ctx context.Context, modelId string, performance int, ram int64, limit int) ([]*WorkerInfo, error) {
	// find all worker that at least one installed model's mode_id is equal modelId
	// sort by gpu ram
	findOptions := options.Find()
	findOptions.SetLimit(int64(limit))
	findOptions.SetSort(bson.D{{"hardware.gpu.ram", 1}})

	selector := bson.M{"model_infos.installed_models.model_id": modelId, "hardware.gpu.performance": bson.M{"$gte": performance}, "hardware.gpu.ram": bson.M{"$gte": ram}}
	cursor, err := d.col.Find(ctx, selector, findOptions)
	if err != nil {
		return nil, err
	}
	defer cursor.Close(ctx)

	var workers []*WorkerInfo
	if err = cursor.All(ctx, &workers); err != nil {
		return nil, err
	}
	return workers, nil
}

func (d *WorkerInfoOperator) FindWorkerByWorkerId(ctx context.Context, workerId string) (*WorkerInfo, error) {
	var worker *WorkerInfo
	err := d.col.FindOne(ctx, bson.M{"worker_id": workerId}).Decode(&worker)
	return worker, err
}
