package operator

import (
	"context"
	crand "crypto/rand"
	"encoding/hex"
	"fmt"
	"github.com/google/uuid"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"log"
	"math/rand"
	"mogo/types"
	"strconv"
	"testing"
	"time"
)

var (
	maxModelId = 10000
	idlist     = make([]string, 0, 1000000)
	//workers    = make([]*DbWorkerInfo, 0, 1000000)
	database                = "test"
	collection              = "workers"
	workerRunningCollection = "worker_running"
)

func ConnectMongoDB(uri string, username, passwd string) (*mongo.Client, error) {
	ctx, _ := context.WithTimeout(context.Background(), 10*time.Second)
	client, err := mongo.Connect(ctx, options.Client().ApplyURI(uri).SetAuth(options.Credential{
		Username: username,
		Password: passwd,
	}))
	if err != nil {
		return nil, err
	}

	return client, nil
}

func init() {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	idlist = initdata(client)
}

func initdata(client *mongo.Client) []string {
	t1 := time.Now()
	db := NewDBWorker(client, database)
	dbRunning := NewDBWorkerRunning(client, database)

	// Insert 1,000,000 DbWorkerInfo to operator
	for i := 0; i < 1000; i++ {
		worker := generateAWroker()
		result, err := db.InsertWorker(context.Background(), worker)
		if err != nil {
			panic(fmt.Sprintf("insert worker failed with err:%s", err))
		}
		{
			// add worker running info to dbRunning
			for _, model := range worker.Models.RunningModels {
				id, _ := strconv.Atoi(model.ModelID)
				runningInfo := &WorkerRunningInfo{
					WorkerId: worker.WorkerId,
					ModelId:  id,
					ExecTime: model.ExecTime,
				}
				_, err := dbRunning.Insert(context.Background(), runningInfo)
				if err != nil {
					panic(fmt.Sprintf("insert worker failed with err:%s", err))
				}
			}
		}

		id, ok := result.InsertedID.(primitive.ObjectID)
		if !ok {
			panic("inserted id is not primitive.ObjectID")
		}
		idlist = append(idlist, id.Hex())
		//fmt.Printf("insert worker %s: %v\n", id.Hex(), worker)
	}
	t2 := time.Now()
	fmt.Printf("init data cost %s\n", t2.Sub(t1).String())
	return idlist
}

func getRandId(max int) string {
	return strconv.Itoa(rand.Intn(max) + 1)
}

func getRandIdInt(max int) int {
	return rand.Intn(max) + 1
}

func generateAWroker() *WorkerInfo {
	return &WorkerInfo{
		WorkerId: uuid.NewString(),
		NodeInfo: generateANodeInfo(),
		Models:   generateAModel(),
		Hardware: generateAHardware(),
	}
}

func generateANodeInfo() *types.NodeInfo {
	x := make([]byte, 32)
	y := make([]byte, 32)
	z := make([]byte, 20)
	crand.Read(x)
	crand.Read(y)
	crand.Read(z)

	return &types.NodeInfo{
		MinerPubkey:    hex.EncodeToString(x) + hex.EncodeToString(y),
		BenefitAddress: hex.EncodeToString(z),
	}
}

func generateACpu() *types.CpuInfo {
	return &types.CpuInfo{
		Model:   "Intel(R) Core(TM) i7-8700 CPU @ 3.20GHz",
		Number:  1,
		Cores:   8,
		Threads: 16,
		Usage:   int(rand.Intn(30) + 40),
	}
}

func generateADisk() *types.DiskInfo {
	return &types.DiskInfo{
		Total: 1024 * 1024 * 1024,
		Free:  100 * 1024 * 1024,
	}
}

func generateANet() *types.NetInfo {
	return &types.NetInfo{
		IP:        fmt.Sprintf("192.168.1.%d", rand.Intn(255)),
		Mac:       fmt.Sprintf("%02x:%02x:%02x:%02x:%02x:%02x", 1, 2, 3, 4, 5, 6),
		Bandwidth: rand.Intn(30) + 60,
	}
}

func generateARam() *types.RamInfo {
	return &types.RamInfo{
		Total: 32 * 1024 * 1024 * 1024,
		Free:  int64(rand.Intn(10) * 1024 * 1024 * 1024),
	}
}

func generateAGpuRam() int64 {
	return 1024 * 1024 * 1024 * int64(rand.Intn(3)*8+8) // 8, 16, 24
}

func generateAGpuModel() string {
	m := rand.Intn(4)*10 + 3060 // 3060 ~ 3090
	return fmt.Sprintf("Nvidia %d", m)
}

func generateAGpuPerformance() int {
	return rand.Intn(100) + 500
}

func generateAIdleGpu(seq int) *types.GpuInfo {
	ram := generateAGpuRam()
	return &types.GpuInfo{
		Seq:         seq,
		UUID:        uuid.NewString(),
		Model:       generateAGpuModel(),
		Performance: generateAGpuPerformance(),
		PowerRating: 100,
		MemTotal:    ram,
		MemFree:     ram,
		Usage:       0,
		Temp:        40,
		PowerRt:     30,
	}
}

func generateAUsageGpu(seq int) *types.GpuInfo {
	ram := generateAGpuRam()
	return &types.GpuInfo{
		Seq:         seq,
		UUID:        uuid.NewString(),
		Model:       generateAGpuModel(),
		Performance: generateAGpuPerformance(),
		PowerRating: 100,
		MemTotal:    ram,
		MemFree:     ram * int64(rand.Intn(3)+3) / 10,
		Usage:       rand.Intn(20) + 70,
		Temp:        40,
		PowerRt:     30,
	}
}

func generateAHardware() *types.HardwareInfo {
	return &types.HardwareInfo{
		CPU: generateACpu(),
		GPU: []*types.GpuInfo{
			generateAIdleGpu(0),
			generateAIdleGpu(1),
			generateAUsageGpu(2),
		},
		RAM:  generateARam(),
		DISK: generateADisk(),
		NET:  generateANet(),
	}
}

func generateAInstallModel() *types.InstalledModel {
	return &types.InstalledModel{
		ModelID:       getRandId(maxModelId),
		DiskSize:      101,
		InstalledTime: time.Now().Unix(),
		LastRunTime:   time.Now().Unix(),
	}
}

func generateARunningModel() *types.RunningModel {
	return &types.RunningModel{
		ModelID:       getRandId(maxModelId),
		GpuSeq:        rand.Intn(3),
		GpuRAM:        generateAGpuRam(),
		StartedTime:   time.Now().Unix(),
		LastWorkTime:  time.Now().Unix(),
		TotalRunCount: rand.Intn(100),
		ExecTime:      rand.Intn(100),
	}
}

func generateAModel() *types.ModelInfo {
	m := &types.ModelInfo{
		InstalledModels: make([]*types.InstalledModel, 0, 1000),
		RunningModels:   make([]*types.RunningModel, 0, 1000),
	}
	for i := 0; i < 500; i++ {
		m.InstalledModels = append(m.InstalledModels, generateAInstallModel())
		if len(m.RunningModels) < 500 {
			m.RunningModels = append(m.RunningModels, generateARunningModel())
		}
	}
	return m
}

func BenchmarkGenerateWorker(b *testing.B) {
	for i := 0; i < b.N; i++ {
		generateAWroker()
	}
}

func BenchmarkDbWorker_InsertWorker(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		b.StopTimer()
		worker := generateAWroker()
		b.StartTimer()
		if _, err := db.InsertWorker(context.Background(), worker); err != nil {
			panic(fmt.Sprintf("insert worker failed with err:%s", err))
		}
	}
}

func BenchmarkDbWorker_InsertWorker_Parallel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			worker := generateAWroker()
			if _, err := db.InsertWorker(context.Background(), worker); err != nil {
				panic(fmt.Sprintf("insert worker failed with err:%s", err))
			}
		}
	})
}

func BenchmarkDbWorker_UpdateHardware(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		b.StopTimer()

		idx := rand.Intn(len(idlist))
		nhardware := generateAHardware()

		b.StartTimer()
		if err := db.UpdateHardware(context.Background(), idlist[idx], nhardware); err != nil {
			panic(fmt.Sprintf("update worker failed with err:%s", err))
		}
	}
}

func BenchmarkDbWorker_UpdateHardware_Parallel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())

	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			idx := rand.Intn(len(idlist))
			nhardware := generateAHardware()
			if err := db.UpdateHardware(context.Background(), idlist[idx], nhardware); err != nil {
				panic(fmt.Sprintf("update worker failed with err:%s", err))
			}
		}
	})
}

func BenchmarkDbWorker_UpdateModel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	b.ResetTimer()

	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	for i := 0; i < b.N; i++ {
		idx := rand.Intn(len(idlist))
		nresource := generateAModel()
		if err := db.UpdateModel(context.Background(), idlist[idx], nresource); err != nil {
			panic(fmt.Sprintf("update worker failed with err:%s", err))
		}
	}
}

func BenchmarkDbWorker_UpdateModel_Parallel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}

	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			idx := rand.Intn(len(idlist))
			nresource := generateAModel()
			if err := db.UpdateModel(context.Background(), idlist[idx], nresource); err != nil {
				panic(fmt.Sprintf("update worker failed with err:%s", err))
			}
		}
	})
}

func BenchmarkDbWorker_UpdateNodeInfo(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	b.ResetTimer()
	nnodeinfo := generateANodeInfo()

	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	for i := 0; i < b.N; i++ {
		idx := rand.Intn(len(idlist))
		if err := db.UpdateNodeInfo(context.Background(), idlist[idx], nnodeinfo); err != nil {
			panic(fmt.Sprintf("update worker failed with err:%s", err))
		}
	}
}

func BenchmarkDbWorker_UpdateNodeInfo_Parallel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	nnodeinfo := generateANodeInfo()
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			idx := rand.Intn(len(idlist))
			if err := db.UpdateNodeInfo(context.Background(), idlist[idx], nnodeinfo); err != nil {
				panic(fmt.Sprintf("update worker failed with err:%s", err))
			}
		}
	})
}

func BenchmarkDbWorker_FindWorkerByInstallModelAndSortByGpuRam(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		installedModelId := getRandId(maxModelId)
		performance := generateAGpuPerformance()
		ram := generateAGpuRam()
		if w, err := db.FindWorkerByInstallModelAndSortByGpuRam(context.Background(), installedModelId, performance, ram, 10); err != nil {
			panic(fmt.Sprintf("find worker failed with err:%s", err))
		} else if len(w) == 0 {
			b.Logf("FindWorkerByInstallModelAndSortByGpuRam find %d with id %s\n", len(w), installedModelId)
		}
	}
}

func BenchmarkDbWorker_FindWorkerByInstallModelAndSortByGpuRam_Parallel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			installedModelId := getRandId(maxModelId)
			performance := generateAGpuPerformance()
			ram := generateAGpuRam()
			if w, err := db.FindWorkerByInstallModelAndSortByGpuRam(context.Background(), installedModelId, performance, ram, 10); err != nil {
				panic(fmt.Sprintf("find worker failed with err:%s", err))
			} else if len(w) == 0 {
				b.Logf("FindWorkerByInstallModelAndSortByGpuRam find %d with id %s\n", len(w), installedModelId)
			}
		}
	})
}

func BenchmarkDbWorker_FindWorkerByRunningModelAndSortByWaitTime(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		runningModelId := getRandId(maxModelId)
		if w, err := db.FindWorkerByRunningModelAndSortByWaitTime(context.Background(), runningModelId, 10); err != nil {
			panic(fmt.Sprintf("find worker failed with err:%s", err))
		} else if len(w) == 0 {
			b.Logf("FindWorkerByRunningModelAndSortByWaitTime find %d with id %s\n", len(w), runningModelId)
		}
	}
}

func BenchmarkDbWorker_FindWorkerByRunningModelAndSortByWaitTime_Parallel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorker(client, database)
	defer db.client.Disconnect(context.Background())
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			runningModelId := getRandId(maxModelId)
			if w, err := db.FindWorkerByRunningModelAndSortByWaitTime(context.Background(), runningModelId, 10); err != nil {
				panic(fmt.Sprintf("find worker failed with err:%s", err))
			} else if len(w) == 0 {
				b.Logf("FindWorkerByRunningModelAndSortByWaitTime find %d with id %s\n", len(w), runningModelId)
			}
		}
	})
}

func BenchmarkDbWorkerRunning_FindWorkerByModelId(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorkerRunning(client, database)
	defer db.client.Disconnect(context.Background())
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		modelId := getRandIdInt(maxModelId)
		if w, err := db.FindWorkerByModelId(context.Background(), modelId, 10); err != nil {
			panic(fmt.Sprintf("find worker failed with err:%s", err))
		} else if len(w) == 0 {
			b.Logf("FindWorkerByModelId find %d with id %d\n", len(w), modelId)
		}
	}
}

func BenchmarkDbWorkerRunning_FindWorkerByModelId_Parallel(b *testing.B) {
	client, err := ConnectMongoDB("mongodb://localhost:27017", "admin", "admin")
	if err != nil {
		log.Fatal(err)
	}
	db := NewDBWorkerRunning(client, database)
	defer db.client.Disconnect(context.Background())
	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			modelId := getRandIdInt(maxModelId)
			if w, err := db.FindWorkerByModelId(context.Background(), modelId, 10); err != nil {
				panic(fmt.Sprintf("find worker failed with err:%s", err))
			} else if len(w) == 0 {
				b.Logf("FindWorkerByModelId find %d with id %d\n", len(w), modelId)
			}
		}
	})
}
