package types

import (
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
)

type NodeInfo struct {
	MinerPubkey    string `bson:"miner_pubkey" json:"miner_pubkey"`
	BenefitAddress string `bson:"benefit_address" json:"benefit_address"`
}

func PbToNodeInfo(pb *omanager.NodeInfo) *NodeInfo {
	return &NodeInfo{
		MinerPubkey:    pb.MinerPubkey,
		BenefitAddress: pb.BenefitAddress,
	}
}

func (n *NodeInfo) ToPb() *omanager.NodeInfo {
	return &omanager.NodeInfo{
		MinerPubkey:    n.MinerPubkey,
		BenefitAddress: n.BenefitAddress,
	}
}

type InstalledModel struct {
	ModelID       string `bson:"model_id" json:"model_id"`
	DiskSize      int64  `bson:"disk_size" json:"disk_size"`
	InstalledTime int64  `bson:"installed_time" json:"installed_time"`
	LastRunTime   int64  `bson:"last_run_time" json:"last_run_time"`
}

func PbToInstalledModel(pb *omanager.InstalledModel) *InstalledModel {
	return &InstalledModel{
		ModelID:       pb.ModelId,
		DiskSize:      pb.DiskSize,
		InstalledTime: pb.InstalledTime,
		LastRunTime:   pb.LastRunTime,
	}
}

func (i *InstalledModel) ToPb() *omanager.InstalledModel {
	return &omanager.InstalledModel{
		ModelId:       i.ModelID,
		DiskSize:      i.DiskSize,
		InstalledTime: i.InstalledTime,
		LastRunTime:   i.LastRunTime,
	}
}

type RunningModel struct {
	ModelID       string `bson:"model_id" json:"model_id"`
	GpuSeq        int    `bson:"gpu_seq" json:"gpu_seq"`
	GpuRAM        int64  `bson:"gpu_ram" json:"gpu_ram"`
	StartedTime   int64  `bson:"started_time" json:"started_time"`
	LastWorkTime  int64  `bson:"last_work_time" json:"last_work_time"`
	TotalRunCount int    `bson:"total_run_count" json:"total_run_count"`
	ExecTime      int    `bson:"exec_time" json:"exec_time"`
}

func PbToRunningModel(pb *omanager.RunningModel) *RunningModel {
	return &RunningModel{
		ModelID:       pb.ModelId,
		GpuSeq:        int(pb.GpuSeq),
		GpuRAM:        pb.GpuRam,
		StartedTime:   pb.StartedTime,
		LastWorkTime:  pb.LastWorkTime,
		TotalRunCount: int(pb.TotalRunCount),
		ExecTime:      int(pb.ExecTime),
	}

}

func (r *RunningModel) ToPb() *omanager.RunningModel {
	return &omanager.RunningModel{
		ModelId:       r.ModelID,
		GpuSeq:        int32(r.GpuSeq),
		GpuRam:        r.GpuRAM,
		StartedTime:   r.StartedTime,
		LastWorkTime:  r.LastWorkTime,
		TotalRunCount: int32(r.TotalRunCount),
		ExecTime:      int32(r.ExecTime),
	}
}

type ModelInfo struct {
	InstalledModels []*InstalledModel `bson:"installed_models" json:"installed_models"`
	RunningModels   []*RunningModel   `bson:"running_models" json:"running_models"`
}

func PbToModelInfo(pb *omanager.ModelsInfo) *ModelInfo {
	installedModels := make([]*InstalledModel, 0, len(pb.InstalledModels))
	for _, m := range pb.InstalledModels {
		installedModels = append(installedModels, PbToInstalledModel(m))
	}
	runningModels := make([]*RunningModel, 0, len(pb.RunningModels))
	for _, m := range pb.RunningModels {
		runningModels = append(runningModels, PbToRunningModel(m))
	}
	return &ModelInfo{
		InstalledModels: installedModels,
		RunningModels:   runningModels,
	}
}

func (m *ModelInfo) ToPb() *omanager.ModelsInfo {
	installedModels := make([]*omanager.InstalledModel, 0, len(m.InstalledModels))
	for _, m := range m.InstalledModels {
		installedModels = append(installedModels, m.ToPb())
	}
	runningModels := make([]*omanager.RunningModel, 0, len(m.RunningModels))
	for _, m := range m.RunningModels {
		runningModels = append(runningModels, m.ToPb())
	}
	return &omanager.ModelsInfo{
		InstalledModels: installedModels,
		RunningModels:   runningModels,
	}
}

type HardwareInfo struct {
	CPU  *CpuInfo   `bson:"CPU" json:"CPU"`
	GPU  []*GpuInfo `bson:"GPU" json:"GPU"`
	RAM  *RamInfo   `bson:"RAM" json:"RAM"`
	DISK *DiskInfo  `bson:"DISK" json:"DISK"`
	NET  *NetInfo   `bson:"NET" json:"NET"`
}

func PbToHardwareInfo(pb *omanager.HardwareInfo) *HardwareInfo {
	cpu := PbToCpuInfo(pb.CPU)
	gpus := make([]*GpuInfo, 0)
	for _, gpu := range pb.GPU {
		gpus = append(gpus, PbToGpuInfo(gpu))
	}
	ram := PbToRamInfo(pb.RAM)
	disk := PbToDiskInfo(pb.DISK)
	net := PbToNetInfo(pb.NET)
	return &HardwareInfo{
		CPU:  cpu,
		GPU:  gpus,
		RAM:  ram,
		DISK: disk,
		NET:  net,
	}
}

func (h *HardwareInfo) ToPb() *omanager.HardwareInfo {
	cpu := h.CPU.ToPb()
	gpus := make([]*omanager.GPUInfo, 0)
	for _, gpu := range h.GPU {
		gpus = append(gpus, gpu.ToPb())
	}
	ram := h.RAM.ToPb()
	disk := h.DISK.ToPb()
	net := h.NET.ToPb()

	return &omanager.HardwareInfo{
		CPU:  cpu,
		GPU:  gpus,
		RAM:  ram,
		DISK: disk,
		NET:  net,
	}
}

type CpuInfo struct {
	Model   string `bson:"model" json:"model "`
	Number  int    `bson:"number" json:"number"`
	Cores   int    `bson:"cores" json:"cores"`
	Threads int    `bson:"threads" json:"threads"`
	Usage   int    `bson:"usage" json:"usage"`
}

func PbToCpuInfo(pb *omanager.CPUInfo) *CpuInfo {
	return &CpuInfo{
		Model:   pb.Model,
		Number:  int(pb.Number),
		Cores:   int(pb.Cores),
		Threads: int(pb.Threads),
		Usage:   int(pb.Usage),
	}
}

func (c *CpuInfo) ToPb() *omanager.CPUInfo {
	return &omanager.CPUInfo{
		Model:   c.Model,
		Number:  int32(c.Number),
		Cores:   int32(c.Cores),
		Threads: int32(c.Threads),
		Usage:   int32(c.Usage),
	}
}

type GpuInfo struct {
	Seq         int    `bson:"seq" json:"seq"`
	UUID        string `bson:"uuid" json:"uuid"`
	Model       string `bson:"model" json:"model "`
	Performance int    `bson:"performance" json:"performance"`
	PowerRating int    `bson:"power_rating" json:"power_rating"`
	MemTotal    int64  `bson:"mem_total" json:"mem_total"`
	MemFree     int64  `bson:"mem_free" json:"mem_free"`
	Usage       int    `bson:"usage" json:"usage"`
	Temp        int    `bson:"temp" json:"temp "`
	PowerRt     int    `bson:"power_rt" json:"power_rt"`
}

func PbToGpuInfo(pb *omanager.GPUInfo) *GpuInfo {
	return &GpuInfo{
		Seq:         int(pb.Seq),
		UUID:        pb.Uuid,
		Model:       pb.Model,
		Performance: int(pb.Performance),
		PowerRating: int(pb.PowerRating),
		MemTotal:    pb.MemTotal,
		MemFree:     pb.MemFree,
		Usage:       int(pb.Usage),
		Temp:        int(pb.Temp),
		PowerRt:     int(pb.PowerRt),
	}
}

func (g *GpuInfo) ToPb() *omanager.GPUInfo {
	return &omanager.GPUInfo{
		Seq:         int32(g.Seq),
		Uuid:        g.UUID,
		Model:       g.Model,
		Performance: int32(g.Performance),
		PowerRating: int32(g.PowerRating),
		MemTotal:    g.MemTotal,
		MemFree:     g.MemFree,
		Usage:       int32(g.Usage),
		Temp:        int32(g.Temp),
		PowerRt:     int32(g.PowerRt),
	}
}

type RamInfo struct {
	Total int64 `bson:"total" json:"total"`
	Free  int64 `bson:"free" json:"free"`
}

func PbToRamInfo(pb *omanager.MemoryInfo) *RamInfo {
	return &RamInfo{
		Total: pb.Total,
		Free:  pb.Free,
	}

}

func (r *RamInfo) ToPb() *omanager.MemoryInfo {
	return &omanager.MemoryInfo{
		Total: r.Total,
		Free:  r.Free,
	}
}

type DiskInfo struct {
	Total int64 `bson:"total" json:"total"`
	Free  int64 `bson:"free" json:"free"`
}

func PbToDiskInfo(pb *omanager.DiskInfo) *DiskInfo {
	return &DiskInfo{
		Total: pb.Total,
		Free:  pb.Free,
	}
}

func (d *DiskInfo) ToPb() *omanager.DiskInfo {
	return &omanager.DiskInfo{
		Total: d.Total,
		Free:  d.Free,
	}
}

type NetInfo struct {
	IP        string `bson:"ip" json:"ip"`
	Mac       string `bson:"mac" json:"mac"`
	Bandwidth int    `bson:"bandwidth" json:"bandwidth"`
}

func PbToNetInfo(pb *omanager.NetInfo) *NetInfo {
	return &NetInfo{
		IP:        pb.Ip,
		Mac:       pb.Mac,
		Bandwidth: int(pb.Bandwidth),
	}
}

func (n *NetInfo) ToPb() *omanager.NetInfo {
	return &omanager.NetInfo{
		Ip:        n.IP,
		Mac:       n.Mac,
		Bandwidth: int32(n.Bandwidth),
	}
}
