package server

import (
	"bytes"
	"context"
	"errors"
	"github.com/ethereum/go-ethereum/common"
	"github.com/google/uuid"
	lru "github.com/hashicorp/golang-lru"
	"github.com/odysseus/mogo/operator"
	"github.com/odysseus/mogo/types"
	"github.com/odysseus/nodemanager/standardlib"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	log "github.com/sirupsen/logrus"
	"golang.org/x/crypto/sha3"
	"strconv"
	"time"
)

type workerUsageInfo struct {
	hwUsage *omanager.HardwareUsage
}

type sendMsgCallback struct {
	msg      *omanager.ManagerMessage
	callback func(err error) bool
}
type Worker struct {
	wm       *WorkerManager
	quit     chan interface{}
	taskCh   chan *dispatchTask
	msgCh    chan *omanager.WorkerMessage
	sendCh   chan sendMsgCallback
	resultCh chan *omanager.SubmitTaskResult

	uuid            int64 // worker uuid in the local.
	heartBeat       int64
	registed        bool // worker is registed to this nm.
	online          bool
	disconnect      bool
	nonce           int
	latestNmValue   string
	addFirstSucceed bool
	lastFreeGpu     *omanager.GPUInfo

	info           *omanager.NodeInfoResponse
	usage          workerUsageInfo
	workerAddr     string // worker address from public-key
	deviceInfoHash []byte
	recentTask     *lru.ARCCache
	status         string
	errCh          chan error
	infoOp         *operator.WorkerInfoOperator
	installOp      *operator.WorkerInstalledOperator
	runningOp      *operator.WorkerRunningOperator

	stream omanager.NodeManagerService_RegisterWorkerServer
}

func (w *Worker) ProfitAccount() common.Address {
	if w.info != nil {
		return common.HexToAddress(w.info.Info.BenefitAddress)
	}
	return common.Address{}
}

func (w *Worker) WorkerAccount() common.Address {
	return common.HexToAddress(w.workerAddr)
}

func (w *Worker) Disconnect() {
	w.disconnect = true
}

func (w *Worker) DisConnected() bool {
	return w.disconnect
}

func (w *Worker) SendToWorker(msg *omanager.ManagerMessage, callback func(err error) bool) {
	w.sendCh <- sendMsgCallback{msg, callback}
}

func (w *Worker) SendMessage() {
	l := log.WithField("worker-uuid", w.uuid)
	defer func() {
		if e := recover(); e != nil {
			l.WithField("worker-addr", w.workerAddr).Error("worker handle message panic")
		}
	}()
	for w.DisConnected() == false {
		select {
		case <-w.quit:
			return
		case m := <-w.sendCh:
			if w.DisConnected() {
				return
			}
			start := time.Now()
			if m.msg.Message != nil {
				err := w.stream.Send(m.msg)
				if err != nil {
					log.WithError(err).Error("send message to worker failed")
				}
				if m.callback != nil {
					m.callback(err)
				}
			}
			log.WithField("worker-addr", w.workerAddr).WithField("send duration", time.Now().Sub(start).String()).Trace("send msg to worker")
		}
	}

}

func (w *Worker) RecvMessage() {
	l := log.WithField("worker-uuid", w.uuid)
	defer func() {
		if e := recover(); e != nil {
			l.WithField("worker-addr", w.workerAddr).Error("worker handle message panic")
		}
	}()
	for {
		start := time.Now()
		wmsg, err := w.stream.Recv()
		if err != nil {
			l.WithError(err).WithField("worker-addr", w.workerAddr).WithField("recv duration", time.Now().Sub(start).String()).Error("recv msg failed")
			w.errCh <- errors.New("recv msg failed")
			return
		}
		if w.DisConnected() {
			return
		}
		w.msgCh <- wmsg
	}
}

func (w *Worker) doGoodBye(msg *omanager.WorkerMessage_GoodbyeMessage) {
	w.online = false
	w.quit <- msg.GoodbyeMessage.Reason
	close(w.taskCh)
}

func (w *Worker) doSubmitAck(msg *omanager.WorkerMessage_SubmitTaskAck) {
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
		"task-id":     msg.SubmitTaskAck.TaskId,
	}).Debug("receive worker submit task ack")
	if v, ok := w.recentTask.Get(msg.SubmitTaskAck.TaskId); ok {
		dtask := v.(*dispatchTask)
		dtask.setAck(msg.SubmitTaskAck)
	} else {
		log.WithFields(log.Fields{
			"worker-addr": w.workerAddr,
			"task-id":     msg.SubmitTaskAck.TaskId,
		}).Warn("ack not found task")
	}
}

func (w *Worker) doSubmitResult(msg *omanager.WorkerMessage_SubmitTaskResult) {
	w.resultCh <- msg.SubmitTaskResult
}

func (w *Worker) doHeartBeat(msg *omanager.WorkerMessage_HeartbeatResponse) {
	w.online = true
	w.heartBeat = time.Now().Unix()
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
		"TTL":         time.Now().Unix() - int64(msg.HeartbeatResponse.Timestamp),
	}).Trace("receive worker heartbeat")
}

func (w *Worker) doUpdateBenefit(msg *omanager.WorkerMessage_BenefitAddrUpdate) {
	if w.info != nil {
		w.info.Info.BenefitAddress = msg.BenefitAddrUpdate.BenefitAddress
	}
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
		"benefit":     msg.BenefitAddrUpdate.BenefitAddress,
	}).Info("receive worker benefit address update")
	// update to mogo.
	w.infoOp.UpdateBenefitAddr(context.TODO(), w.workerAddr, msg.BenefitAddrUpdate.BenefitAddress)
}

func (w *Worker) doGetNodeInfo(msg *omanager.WorkerMessage_NodeInfo) {
	if w.info == nil {
		w.info = &omanager.NodeInfoResponse{
			Info:     msg.NodeInfo.Info,
			Hardware: msg.NodeInfo.Hardware,
			Models:   msg.NodeInfo.Models,
		}
	} else {
		// todo: do update ?
	}
}

func (w *Worker) doFetchStdTask(msg *omanager.WorkerMessage_FetchStandardTask) {
	if w.info == nil {
		return
	}
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
	}).Debugf("receive worker fetch std task request:%v", msg.FetchStandardTask.TaskType)
	pushTask := standardlib.StdTask{}
	task, exist := w.wm.std.GetTask(msg.FetchStandardTask.TaskType)
	if exist {
		stdlib := w.wm.std.GetStdLib(task.TaskType)
		if stdlib == nil {
			log.WithField("task-type", task.TaskType).Warn("not found std lib")
			return
		}
		pushTask = task
		pushTask.TaskId = uuid.NewString()
		param, err := stdlib.GenerateParam(0)
		if err != nil {
			log.WithError(err).WithField("task-type", task.TaskType).Error("generate param failed")
			return
		}
		pushTask.TaskParam = []byte(param)
		pushTask.TaskInLen = int32(len(param))
		pushTask.TaskUid = "0"
		pushTask.TaskTimestamp = uint64(time.Now().UnixNano())
		pushTask.TaskKind = odysseus.TaskKind_StandardTask
		pushTask.TaskFee = "0"
		dtask := newDispatchTask(w, &pushTask.TaskContent)
		w.taskCh <- dtask
	} else {
		log.WithField("task-type", msg.FetchStandardTask.TaskType).Warn("not found std task")
	}
}

func (w *Worker) doGetDeviceInfo(msg *omanager.WorkerMessage_DeviceInfo) {
	if w.info == nil {
		return
	}
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
	}).Debugf("receive worker device info:%v", msg.DeviceInfo)
	{
		var infoHash [32]byte
		infoHash = sha3.Sum256([]byte(msg.DeviceInfo.String()))
		if bytes.Compare(infoHash[:], w.deviceInfoHash) != 0 && w.registed {
			// check device info changed, and update to mogo.
			w.infoOp.UpdateHardware(context.TODO(), w.workerAddr, types.PbToHardwareInfo(msg.DeviceInfo.Hardware))
		}
		// update local cache.
		w.deviceInfoHash = infoHash[:]
		w.info.Hardware = msg.DeviceInfo.Hardware
	}
}

func (w *Worker) doDeviceUsage(msg *omanager.WorkerMessage_DeviceUsage) error {
	w.usage.hwUsage = msg.DeviceUsage.Usage
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
	}).Debugf("receive worker device usage:%v", msg.DeviceUsage.Usage)

	if !w.registed {
		return nil
	}
	// 1. update usage to hardware mogo.
	return w.infoOp.UpdateHardwareUsage(context.TODO(), w.workerAddr, types.PbToDeviceUsage(msg.DeviceUsage.Usage))
}

func (w *Worker) doAddRunningModel(msg *omanager.WorkerMessage_AddModelRunning) {
	if !w.registed {
		return
	}
	models := make([]*types.RunningModel, 0)
	log.WithFields(log.Fields{
		"worker-addr":    w.workerAddr,
		"running-models": msg.AddModelRunning.Models,
		"count":          len(msg.AddModelRunning.Models),
	}).Debugf("receive worker add running models")

	for _, model := range msg.AddModelRunning.Models {
		models = append(models, types.PbToRunningModel(model))
	}
	// 1. update model to worker info mogo.
	err := w.infoOp.AddRunningModels(context.TODO(), w.workerAddr, models)
	if err != nil {
		log.WithError(err).Error("add worker running model failed")
		return
	}

	// 2. add info to worker running model.
	running := make([]*operator.WorkerRunningInfo, len(models))
	for i, model := range models {
		id, _ := strconv.Atoi(model.ModelID)
		running[i] = &operator.WorkerRunningInfo{

			WorkerId: w.workerAddr,
			ModelId:  id,
			ExecTime: model.ExecTime,
		}
	}
	_, err = w.runningOp.InsertMany(context.TODO(), running)
	if err != nil {
		log.WithError(err).Error("add worker running info failed")
	}
}

func (w *Worker) doRemoveRunningModel(msg *omanager.WorkerMessage_DelModeRunning) {
	if !w.registed {
		return
	}
	models := make([]int, 0)
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
		"count":       len(msg.DelModeRunning.ModelIds),
	}).Debugf("receive worker remove running model:%v", msg.DelModeRunning.ModelIds)

	for _, model := range msg.DelModeRunning.ModelIds {
		id, _ := strconv.Atoi(model)
		models = append(models, id)
	}
	// 1. remove model from worker info mogo.
	err := w.infoOp.DeleteRunningModels(context.TODO(), w.workerAddr, models)
	if err != nil {
		log.WithError(err).Error("remove worker running model failed")
		return
	}

	// 2. remove info from worker running model.
	_, err = w.runningOp.DeleteMany(context.TODO(), w.workerAddr, models)
	if err != nil {
		log.WithError(err).Error("remove worker running info failed")
	}
}

func (w *Worker) doAddInstalledModel(msg *omanager.WorkerMessage_AddModelInstalled) {
	if !w.registed {
		return
	}
	models := make([]*types.InstalledModel, 0)
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
		"count":       len(msg.AddModelInstalled.Models),
	}).Debugf("receive worker add installed model:%v", msg.AddModelInstalled.Models)

	for _, model := range msg.AddModelInstalled.Models {
		models = append(models, types.PbToInstalledModel(model))
	}
	// 1. update model to worker info mogo.
	err := w.infoOp.AddInstalledModels(context.TODO(), w.workerAddr, models)
	if err != nil {
		log.WithError(err).Error("add worker installed model failed")
		return
	}

	maxGpuFree := w.info.Hardware.GPU[0]
	for _, gpu := range w.info.Hardware.GPU {
		if gpu.MemFree > maxGpuFree.MemFree {
			maxGpuFree = gpu
		}
	}

	// 2. add info to worker installed model.
	installed := make([]*operator.WorkerInstalledInfo, len(models))
	for i, model := range models {
		id, _ := strconv.Atoi(model.ModelID)
		installed[i] = &operator.WorkerInstalledInfo{
			WorkerId: w.workerAddr,
			ModelId:  id,
			GpuFree:  maxGpuFree.MemFree,
			GpuSeq:   int(maxGpuFree.Seq),
		}
	}

	_, err = w.installOp.InsertMany(context.TODO(), installed)
	if err != nil {
		log.WithError(err).Error("add worker installed info failed")
	}

}

func (w *Worker) doRemoveInstalledModel(msg *omanager.WorkerMessage_DelModelInstalled) {
	//if !w.registed {
	//	return
	//}
	models := make([]int, 0)
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
		"count":       len(msg.DelModelInstalled.ModelIds),
	}).Debugf("receive worker remove installed model:%v", msg.DelModelInstalled.ModelIds)

	for _, model := range msg.DelModelInstalled.ModelIds {
		id, _ := strconv.Atoi(model)
		models = append(models, id)
	}
	// 1. remove model from worker info mogo.
	err := w.infoOp.DeleteInstalledModels(context.TODO(), w.workerAddr, models)
	if err != nil {
		log.WithError(err).Error("remove worker installed model failed")
		return
	}

	// 2. remove info from worker installed model.
	_, err = w.installOp.DeleteMany(context.TODO(), w.workerAddr, models)
	if err != nil {
		log.WithError(err).Error("remove worker installed info failed")
	}
}

func (w *Worker) doInstalledModelStatus(msg *omanager.WorkerMessage_InstalledModelStatus) {
	//if !w.registed {
	//	return
	//}
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
	}).Debugf("receive worker installed model status:%v", msg.InstalledModelStatus)
	// 1. update model status to worker info mogo.
	id, _ := strconv.Atoi(msg.InstalledModelStatus.ModelId)
	err := w.infoOp.UpdateInstalledModelStatus(context.TODO(), w.workerAddr, id, msg.InstalledModelStatus.LastRunTime)
	if err != nil {
		log.WithError(err).Error("update worker installed model status failed")
	}
}

func (w *Worker) doRunningModelStatus(msg *omanager.WorkerMessage_RunningModelStatus) {
	//if !w.registed {
	//	return
	//}
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
	}).Debugf("receive worker running model status:%v", msg.RunningModelStatus)
	// 1. update model status to worker info mogo.
	id, _ := strconv.Atoi(msg.RunningModelStatus.ModelId)
	err := w.infoOp.UpdateRunningModelStatus(context.TODO(), w.workerAddr, id, msg.RunningModelStatus.LastWorkTime,
		int64(msg.RunningModelStatus.TotalRunCount), int(msg.RunningModelStatus.ExecTime))
	if err != nil {
		log.WithError(err).Error("update worker running model status failed")
	}
	// 2. update model exec time to worker running model.
	err = w.runningOp.UpdateExecTime(context.TODO(), w.workerAddr, id, int(msg.RunningModelStatus.ExecTime))
	if err != nil {
		log.WithError(err).Error("update worker running model exec time failed")
	}
}

func (w *Worker) doGPUUsage(msg *omanager.WorkerMessage_GpuUsage) {
	if w.usage.hwUsage == nil {
		return
	}
	log.WithFields(log.Fields{
		"worker-addr": w.workerAddr,
		"gpu-usage":   msg.GpuUsage,
	}).Debug("receive worker gpu usage")
	// 1. update gpu usage to hardware mogo.
	gpuusages := make([]types.GpuUsage, 0)
	for _, gpu := range msg.GpuUsage.Usages {
		gpuusages = append(gpuusages, types.PbToGpuUsage(gpu))
	}
	err := w.infoOp.UpdateGPUUsage(context.TODO(), w.workerAddr, gpuusages)
	if err != nil {
		log.WithError(err).Error("update worker gpu usage failed")
	}
	// 2. update gpu usage to worker usage.
	for _, usage := range msg.GpuUsage.Usages {
		for _, gpu := range w.info.Hardware.GPU {
			if gpu.Seq == usage.Seq {
				gpu.Usage = usage.Usage
				gpu.MemFree = usage.MemFree
				gpu.PowerRt = usage.PowerRt
				gpu.Temp = usage.Temp
			}
		}
	}
	// 3. get max gpu free.
	maxFree := w.getMaxGPUFree()
	if w.lastFreeGpu != nil && maxFree.Seq == w.lastFreeGpu.Seq && maxFree.MemFree == w.lastFreeGpu.MemFree {
		// no need update
		return
	} else {
		log.WithFields(log.Fields{
			"worker-addr": w.workerAddr,
			"gpu-free":    maxFree.MemFree,
			"gpu-sed":     maxFree.Seq,
		}).Debug("update worker gpu free")
		w.installOp.UpdateGpuFree(context.TODO(), w.workerAddr, int64(maxFree.MemFree), int(maxFree.Seq))
		w.lastFreeGpu = maxFree
	}

}

func (w *Worker) getMaxGPUFree() *omanager.GPUInfo {
	maxGpuFree := w.info.Hardware.GPU[0]
	for _, gpu := range w.info.Hardware.GPU {
		if gpu.MemFree > maxGpuFree.MemFree {
			maxGpuFree = gpu
		}
	}
	return maxGpuFree
}

func (w *Worker) ModelOperate(info interface{}, operate omanager.ModelOperateType) *omanager.ManagerMessage_ModelOperateRequest {
	request := &omanager.ManagerMessage_ModelOperateRequest{
		ModelOperateRequest: &omanager.ModelOperateRequest{
			ModelOperates: []*omanager.ModelOperate{
				{
					ModelId:   "",
					ImageName: "",
					Username:  "",
					Password:  "",
					Cmd:       "",
					Operate:   operate,
				},
			},
		},
	}
	return request
}
