package server

import (
	"encoding/json"
	"fmt"
	"github.com/odysseus/service-registry/common"
	"github.com/odysseus/service-registry/query"
	"github.com/odysseus/service-registry/registry"
	"math/rand"
	"time"
)

var (
	wr registry.Register = workerRegistry{}
)

type workerRegistry struct {
	worker *Worker
	wm     *WorkerManager
}

func (w workerRegistry) ServiceType() common.ServiceType {
	return common.SERVICE_WORKER
}

func (w workerRegistry) Instance() string {
	return fmt.Sprintf("%s", w.worker.workerAddr)
}

func (w workerRegistry) Status() string {
	return fmt.Sprintf("%s", w.worker.status)
}
func (w workerRegistry) DetailInfo() (json.RawMessage, error) {
	if w.worker == nil {
		return nil, fmt.Errorf("worker is nil")
	}
	if w.worker.workerAddr == "" {
		return nil, fmt.Errorf("worker address is empty")
	}

	taskList := make([]string, 0)
	keys := w.worker.recentTask.Keys()
	for _, key := range keys {
		if d, exist := w.worker.recentTask.Get(key); exist {
			dtask := d.(*dispatchTask)
			if dtask.status < TASK_FINISHED {
				taskList = append(taskList, key.(string))
			}
		}
	}

	info := query.WorkerInfo{
		ActiveNM: make([]string, 0),
		TaskList: taskList,
	}
	if w.worker.info != nil {
		info.BenefitAddress = w.worker.info.Info.BenefitAddress
		info.IP = w.worker.info.Hardware.NET.Ip
	}

	nmList, _ := w.wm.WorkerNmList(w.worker)
	for _, nm := range nmList {
		endpoint, _ := w.wm.parseWorkerNmValue(nm)
		if endpoint != "" {
			info.ActiveNM = append(info.ActiveNM, endpoint)
		}
	}
	info.HearBeat = w.worker.heartBeat * 1000 // to ms
	info.MinerAddress = w.worker.workerAddr
	info.Nonce = 0
	if w.worker.info != nil {
		info.CpuModel = w.worker.info.Hardware.CPU.Model
		info.CpuCore = int(w.worker.info.Hardware.CPU.Cores)
		info.GPUModel = w.worker.info.Hardware.GPU[0].Model
		info.GPURam = int(w.worker.info.Hardware.GPU[0].MemTotal)

		info.RamTotal = w.worker.info.Hardware.RAM.Total
		info.RamUsed = info.RamTotal - w.worker.info.Hardware.RAM.Free
	}
	info.MacAddress = w.worker.info.Hardware.NET.Mac

	// todo: record the create time and bootup time
	info.CreateTime = time.Now().Add(time.Hour * 24 * 4).Unix()
	info.BootupTime = time.Now().Add(time.Hour * 12).Unix()
	info.Workload = rand.Intn(500) + 1000

	return json.Marshal(info)
}
