package nmregistry

import (
	"context"
	"crypto/ecdsa"
	"fmt"
	"github.com/odysseus/nodemanager/config"
	"github.com/odysseus/nodemanager/utils"
	"github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"sync"
	"time"
)

// nmregistry implement register nodemanger to redis, and get nodemanger list from redis.
// filter some nodemanager to worker.

type RegistryInfo struct {
	Pubkey    string `redis:"pubkey"`
	Timestamp int64  `redis:"timestamp"`
	Endpoint  string `redis:"endpoint"`
	Addr      string `redis:"addr"`
}

type NodeManagerInfo struct {
	Pubkey   string `json:"pubkey"`
	Endpoint string `json:"endpoint"`
}

type RegistryService struct {
	nodelist []RegistryInfo
	rdb      *redis.Client
	conf     *config.Config
	rw       sync.RWMutex
	public   ecdsa.PublicKey
	quit     chan struct{}
}

func NewRegistryService(conf *config.Config, rdb *redis.Client, public ecdsa.PublicKey) *RegistryService {
	return &RegistryService{
		rdb:    rdb,
		conf:   conf,
		public: public,
		quit:   make(chan struct{}),
	}
}

func (s *RegistryService) Start() {
	ticker := time.NewTicker(time.Second * 20)
	defer ticker.Stop()
	refresh := time.NewTicker(time.Second * 5)
	defer refresh.Stop()

	for {
		select {
		case <-s.quit:
			return
		case <-ticker.C:
			if err := s.registry(s.rdb); err != nil {
				log.WithError(err).Error("registry failed")
			}
		case <-refresh.C:
			if nodes, err := s.allNodeManager(s.rdb); err != nil {
				log.WithError(err).Error("refresh all nodemanager failed")
			} else {
				s.rw.Lock()
				s.nodelist = nodes
				s.rw.Unlock()
				refresh.Reset(time.Second * 60)
			}

		}
	}
}

func (s *RegistryService) Stop() {
	close(s.quit)
}

func (s *RegistryService) registry(rdb *redis.Client) error {
	priv, err := utils.HexToPrivatekey(s.conf.PrivateKey)
	if err != nil {
		panic(fmt.Sprintf("invalid private key: %s", err))
	}

	addr := utils.PrivatekeyToHex(priv)
	pubHex := utils.PubkeyToHex(&priv.PublicKey)

	err = rdb.HSet(context.Background(), config.NODE_MANAGER_SET+addr, RegistryInfo{
		Pubkey:    pubHex,
		Timestamp: time.Now().Unix(),
		Endpoint:  s.conf.Endpoint,
		Addr:      addr,
	}).Err()
	return err
}

func (s *RegistryService) GetNodeManagerList(filter ManagerFilter) []NodeManagerInfo {
	s.rw.RLock()
	defer s.rw.RUnlock()
	var ret []NodeManagerInfo

	for _, node := range s.nodelist {
		if filter == nil || filter.Filter(node) {
			ret = append(ret, NodeManagerInfo{
				Pubkey:   node.Pubkey,
				Endpoint: node.Endpoint,
			})
		}
	}
	return ret
}

func (s *RegistryService) allNodeManager(rdb *redis.Client) ([]RegistryInfo, error) {
	var ret []RegistryInfo
	var tsExpired = 30
	keys, err := rdb.Keys(context.Background(), config.NODE_MANAGER_SET+"*").Result()
	if err != nil {
		return nil, err
	}
	for _, key := range keys {
		res := rdb.HGetAll(context.Background(), key)
		var info RegistryInfo
		if err := res.Scan(&info); err != nil {
			continue
		}
		if time.Now().Unix()-info.Timestamp > int64(tsExpired) {
			// heart beat expired, ignore this nodemanager
			continue
		}
		ret = append(ret, info)
	}
	return ret, nil
}
