package server

import (
	"crypto/ecdsa"
	"crypto/rand"
	"github.com/odysseus/nodemanager/config"
	"github.com/odysseus/nodemanager/nmregistry"
	"github.com/odysseus/nodemanager/utils"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"google.golang.org/grpc"
	"net"
)

type Node struct {
	registry  *nmregistry.RegistryService
	apiServer *grpc.Server
	rdb       *redis.Client
	wm        *WorkerManager
	privk     *ecdsa.PrivateKey
}

func NewNode() *Node {
	redisConfig := config.GetConfig().Redis
	rdb := utils.NewRedisClient(utils.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	})
	privk, err := utils.HexToPrivatekey(config.GetConfig().PrivateKey)
	if err != nil {
		log.WithError(err).Error("failed to parse node manager private key")
		return nil
	}
	node := &Node{
		rdb:       rdb,
		privk:     privk,
		apiServer: grpc.NewServer(grpc.MaxSendMsgSize(1024*1024*20), grpc.MaxRecvMsgSize(1024*1024*20)),
		registry:  nmregistry.NewRegistryService(config.GetConfig(), rdb, privk.PublicKey),
	}

	node.wm = NewWorkerManager(rdb, node)

	return node
}

func (n *Node) Sign(hash []byte) ([]byte, error) {
	return n.privk.Sign(rand.Reader, hash, nil)
}

func (n *Node) Start() error {
	go n.registry.Start()
	if err := n.apiStart(); err != nil {
		return err
	}
	return nil
}

func (n *Node) apiStart() error {
	lis, err := net.Listen("tcp", config.GetConfig().Endpoint)
	if err != nil {
		log.WithError(err).Error("failed to listen endpoint")
		return err
	}

	omanager.RegisterNodeManagerServiceServer(n.apiServer, &NodeManagerService{
		quit: make(chan struct{}),
		node: n,
	})

	err = n.apiServer.Serve(lis)
	if err != nil {
		log.WithError(err).Error("failed to serve apiserver")
		return err
	}
	log.Info("api started")
	return nil
}

func (n *Node) Stop() {
	n.registry.Stop()
	n.apiServer.Stop()
}
