package server

import (
	"context"
	"errors"
	"github.com/odysseus/nodemanager/utils"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"strconv"
)

var (
	ErrParams    = errors.New("invalid request")
	ErrUnSupport = errors.New("unsupport feature")
)

type NodeManagerService struct {
	node *Node
	quit chan struct{}
	omanager.UnimplementedNodeManagerServiceServer
}

func (n *NodeManagerService) ManagerList(ctx context.Context, request *omanager.ManagerListRequest) (*omanager.ManagerListResponse, error) {
	list := n.node.registry.GetNodeManagerList(nil)
	res := new(omanager.ManagerListResponse)
	res.Managers = make([]omanager.NodeManagerInfo, 0, len(list))
	for _, v := range list {
		res.Managers = append(res.Managers, omanager.NodeManagerInfo{
			Publickey: v.Pubkey,
			Endpoint:  v.Endpoint,
		})
	}
	return res, nil
}

func (n *NodeManagerService) RegisterWorker(client omanager.NodeManagerService_RegisterWorkerServer) error {
	uuid := utils.GetSnowflakeId()
	worker, err := n.node.wm.AddNewWorker(uuid, client)
	if err != nil {
		return err
	}
	return n.node.wm.manageWorker(worker)
}

func (n *NodeManagerService) DispatchTask(ctx context.Context, request *omanager.DispatchTaskRequest) (*omanager.DispatchTaskResponse, error) {
	if request == nil {
		return nil, errors.New("invalid request")
	}
	uuid, err := strconv.ParseInt(request.Miner, 10, 64)
	if err != nil {
		return nil, errors.New("not found worker")
	}
	worker := n.node.wm.GetWorker(uuid)
	dtask := &dispatchTask{
		task:  request.TaskData,
		errCh: make(chan error, 1),
	}

	worker.taskCh <- dtask

	// wait task send to worker.
	if err, _ := <-dtask.errCh; err != nil {
		return nil, err
	}

	res := new(omanager.DispatchTaskResponse)
	res.TaskId = request.TaskData.TaskId
	res.Miner = request.Miner
	return res, nil
}
