package config

import (
	"fmt"
	"github.com/BurntSushi/toml"
	"github.com/odysseus/nodemanager/utils"
	log "github.com/sirupsen/logrus"
	"io/ioutil"
)

type MysqlConfig struct {
	Host   string `json:"host" toml:"host"`
	Port   int    `json:"port" toml:"port"`
	User   string `json:"user" toml:"user"`
	Passwd string `json:"password" toml:"password"`
	DbName string `json:"database" toml:"database"`
}

type KafkaConfig struct {
	Brokers      string `json:"brokers" toml:"brokers"`
	ReceiptTopic string `json:"receipt_topic" toml:"receipt_topic"`
	TaskTopic    string `json:"task_topic" toml:"task_topic"`
	ProofTopic   string `json:"proof_topic" toml:"proof_topic"`
}

type RedisConfig struct {
	Addr     string `json:"addr" toml:"addr"`
	Password string `json:"password" toml:"password"`
	DbIndex  int    `json:"db_index" toml:"db_index"`
}

type MongoDbConfig struct {
	Url      string `json:"url" toml:"url"`
	Database string `json:"database" toml:"database"`
	User     string `json:"user" toml:"user"`
	Passwd   string `json:"password" toml:"password"`
}

type TickerConfig struct {
	HeartBeat              int `json:"heart_beat" toml:"heart_beat"`
	StatusTicker           int `json:"status_ticker" toml:"status_ticker"`
	DeviceInfoTicker       int `json:"device_info_ticker" toml:"device_info_ticker"`
	DeviceUsageTicker      int `json:"device_usage_ticker" toml:"device_usage_ticker"`
	WorkerTaskExpireTicker int `json:"worker_task_expire_ticker" toml:"worker_task_expire_ticker"`
}

type Config struct {
	PrivateKey       string `json:"private_key" toml:"private_key"`
	RemoteHost       string `json:"remote_host" toml:"remote_host"`
	LocalHost        string `json:"local_host" toml:"local_host"`
	Port             int    `json:"port" toml:"port"`
	StandardTaskFile string `json:"standard_task_file" toml:"standard_task_file"`
	//Endpoint   string       `json:"endpoint" toml:"endpoint"`
	MetricPort                 int           `json:"metrics_port" toml:"metrics_port"`
	EnablePay                  bool          `json:"enable_pay" toml:"enable_pay"`
	WorkerMultiple             int           `json:"worker_multiple" toml:"worker_multiple"`
	WorkerSignatureExpiredTime int64         `json:"worker_signature_expired_time" toml:"worker_signature_expired_time"`
	Redis                      RedisConfig   `json:"redis" toml:"redis"`
	DbConfig                   MysqlConfig   `json:"mysql" toml:"mysql"`
	Tickers                    TickerConfig  `json:"ticker" toml:"ticker"`
	Kafka                      KafkaConfig   `json:"kafka" toml:"kafka"`
	Mongo                      MongoDbConfig `json:"mongodb" toml:"mongodb"`
	ModelInfoUrl               string        `json:"model_info_url" toml:"model_info_url"`
}

var _cfg *Config = nil

func ParseConfig(path string) (*Config, error) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		log.Error("get config failed", "err", err)
		panic(err)
	}
	err = toml.Unmarshal(data, &_cfg)
	// err = json.Unmarshal(data, &_cfg)
	if err != nil {
		log.Error("unmarshal config failed", "err", err)
		panic(err)
	}
	if _cfg.RemoteHost == "" {
		ip, err := utils.PublicIp()
		if err != nil {
			log.Error("get public ip failed", "err", err)
		}
		_cfg.RemoteHost = ip
	}
	return _cfg, nil
}

func GetConfig() *Config {
	return _cfg
}
func (conf *Config) PublicEndpoint() string {
	return fmt.Sprintf("%s:%d", conf.RemoteHost, conf.Port)
}

func (conf *Config) ApiEndpoint() string {
	return fmt.Sprintf("0.0.0.0:%d", conf.Port)
}

func (conf *Config) StandardTaskFilePath() string {
	return conf.StandardTaskFile
}

func (conf *Config) GetWorkerMultiple() int {
	if conf.WorkerMultiple > 0 {
		return conf.WorkerMultiple
	} else {
		return 1
	}
}

func (conf *Config) GetWorkerSignatureExpiredTime() int64 {
	if conf.WorkerSignatureExpiredTime < 1 {
		return 1
	}
	return conf.WorkerSignatureExpiredTime
}

func (conf *Config) GetModelInfoUrl() string {
	return conf.ModelInfoUrl
}
