package distribute

import (
	"encoding/json"
	"github.com/odysseus/nodemanager/config"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"net/http"
	"sync"
	"time"
)

type Distributor struct {
	workers  sync.Map
	modelLib ModelLibrary
	manager  WorkerManager
}

func StartDistributor(manager WorkerManager) *Distributor {
	infos, _ := getModelInfo(config.GetConfig().GetModelInfoUrl())
	lib := NewHeapModelInfos(infos)
	dis := &Distributor{
		modelLib: lib,
		manager:  manager,
	}
	go dis.loop()

	return dis
}

func (d *Distributor) AddWorker(id int64, addr string, info *omanager.NodeInfoResponse) {
	if _, exist := d.workers.Load(id); exist {
		return
	}
	worker := NewImageWorker(addr, d.modelLib, d.manager)
	d.workers.Store(info.Info.MinerPubkey, worker)
	go worker.DistributeImages()
}

func (d *Distributor) RemoveWorker(id int64) {
	if w, exist := d.workers.Load(id); exist {
		w.(*imageWorker).Exit()
	}
	d.workers.Delete(id)
}

func (d *Distributor) UpdateWorkerInfo(id int64, info *omanager.NodeInfoResponse) {
	if w, exist := d.workers.Load(id); exist {
		w.(*imageWorker).UpdateInfo(info)
	}
}

func (d *Distributor) loop() {
	tc := time.NewTicker(time.Minute * 10)
	defer tc.Stop()

	for {
		select {
		case <-tc.C:
			infos, _ := getModelInfo(config.GetConfig().GetModelInfoUrl())
			d.modelLib.UpdateModelInfo(infos)
		}
	}
}

var (
	hclient = &http.Client{Timeout: 10 * time.Second}
)

func getModelInfo(url string) ([]ModelDetailInfo, error) {
	r, err := hclient.Get(url)
	if err != nil {
		return []ModelDetailInfo{}, err
	}
	defer r.Body.Close()
	data := struct {
		Infos []ModelDetailInfo `json:"data"`
	}{}
	if err := json.NewDecoder(r.Body).Decode(&data); err != nil {
		return []ModelDetailInfo{}, err
	}
	return data.Infos, nil
}
