package distribute

import (
	"encoding/json"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"golang.org/x/crypto/sha3"
	"sort"
	"strconv"
	"sync"
	"time"
)

type imageWorker struct {
	addr         string
	modelLibrary ModelLibrary
	manager      WorkerManager
	mux          sync.Mutex
	info         *omanager.NodeInfoResponse
	quit         chan struct{}
}

func NewImageWorker(addr string, modellib ModelLibrary, manager WorkerManager) *imageWorker {
	return &imageWorker{
		addr:         addr,
		modelLibrary: modellib,
		manager:      manager,
		quit:         make(chan struct{}),
	}
}

func (w *imageWorker) Exit() {
	defer func() {
		recover()
	}()
	close(w.quit)
}

func (w *imageWorker) UpdateInfo(info *omanager.NodeInfoResponse) {
	w.mux.Lock()
	defer w.mux.Unlock()
	w.info = info
}

func (w *imageWorker) getInfo() omanager.NodeInfoResponse {
	w.mux.Lock()
	defer w.mux.Unlock()
	return *w.info
}

// Seed return the seed of the worker.
func (w *imageWorker) Seed() []byte {
	info := w.getInfo()
	return []byte(info.Info.MinerPubkey)
}

// IsInstalled check if the model is installed on worker.
func (w *imageWorker) IsInstalled(modelId int) bool {
	info := w.getInfo()
	for _, v := range info.Models.InstalledModels {
		id, _ := strconv.Atoi(v.ModelId)
		if id == modelId {
			return true
		}
	}
	return false
}

// IsRunning check if the model is running on the worker.
func (w *imageWorker) IsRunning(modelId int) bool {
	info := w.getInfo()
	for _, v := range info.Models.RunningModels {
		id, _ := strconv.Atoi(v.ModelId)
		if id == modelId {
			return true
		}
	}
	return false
}

// CanInstall check if the worker can install the model.
func (w *imageWorker) CanInstall(model ModelDetailInfo) bool {
	info := w.getInfo()
	diskFree := info.Hardware.DISK.Free
	pending := int64(0)
	for _, v := range info.Models.WaitToInstallModels {
		id, _ := strconv.Atoi(v.ModelId)
		minfo := w.modelLibrary.FindModel(id)
		pending += minfo.HardwareRequire.IntDiskSize()
	}
	diskPass := (diskFree - pending) >= model.HardwareRequire.IntDiskSize()
	gpuPass := false
	{
		gpuList := w.info.Hardware.GPU
		require := model.HardwareRequire
		minGpuRequire := require.IntGpu(0)
		for i, _ := range require.Gpus {
			if require.IntGpu(i) < minGpuRequire {
				minGpuRequire = require.IntGpu(i)
			}
		}

		for _, v := range gpuList {
			if v.Performance >= int32(minGpuRequire) && v.MemTotal >= require.IntMemorySize() {
				gpuPass = true
				break
			}
		}
	}
	return diskPass && gpuPass
}

// CanRun check if the worker can run the model with no stop other models.
func (w *imageWorker) CanRun(model ModelDetailInfo) bool {
	info := w.getInfo()
	// check gpu memory and performance
	gpuList := info.Hardware.GPU
	require := model.HardwareRequire
	minGpuRequire := require.IntGpu(0)
	for i, _ := range require.Gpus {
		if require.IntGpu(i) < minGpuRequire {
			minGpuRequire = require.IntGpu(i)
		}
	}

	gpuPass := false
	for _, v := range gpuList {
		if v.Performance >= int32(minGpuRequire) && v.MemFree >= require.IntMemorySize() {
			gpuPass = true
			break
		}
	}
	return gpuPass
}

// CanForceRun check if the worker can run the model but need stop other models.
func (w *imageWorker) CanForceRun(model ModelDetailInfo) bool {
	info := w.getInfo()
	// check gpu memory and performance
	gpuList := info.Hardware.GPU
	require := model.HardwareRequire
	minGpuRequire := require.IntGpu(0)
	for i, _ := range require.Gpus {
		if require.IntGpu(i) < minGpuRequire {
			minGpuRequire = require.IntGpu(i)
		}
	}

	gpuPass := false
	for _, v := range gpuList {
		if v.Performance >= int32(minGpuRequire) && v.MemTotal >= require.IntMemorySize() {
			gpuPass = true
			break
		}
	}
	return gpuPass
}

func (w *imageWorker) DistributeImages() {
	tc := time.NewTicker(time.Second)
	defer tc.Stop()

	for {
		select {
		case <-w.quit:
			return
		case <-tc.C:
			w.distribute()
			tc.Reset(time.Minute * 20)
		}
	}

}

type DistributeMode int

const (
	GreedyMode  DistributeMode = iota // 从热度由高到低，尽可能安装更多的模型
	HashingMode                       // 从热度由高到低，选择相匹配的模型进行安装
)

func (w *imageWorker) getOp(model ModelDetailInfo, opType omanager.ModelOperateType) *omanager.ModelOperate {
	cmdstr, _ := json.MarshalIndent(model.Cmd, "", "    ")
	op := new(omanager.ModelOperate)
	op.Operate = opType
	op.ModelId = strconv.FormatInt(int64(model.TaskID), 10)
	op.ImageName = model.ImageName
	op.Username = ""
	op.Password = ""
	op.Cmd = string(cmdstr)
	op.GpuSeq = 0

	return op
}

func (w *imageWorker) distributeMatch(mode DistributeMode, model ModelDetailInfo, hash []byte) bool {
	if mode == GreedyMode {
		return true
	}
	if mode == HashingMode {
		level := w.modelLibrary.GetModelUsedLevel(model.TaskID)
		weights := 0
		switch level {
		case ModelUsedLevelSuperLow:
			weights += 2
		case ModelUsedLevelVeryLow:
			weights += 5
		case ModelUsedLevelLow:
			weights += 10
		case ModelUsedLevelMiddle:
			weights += 30
		case ModelUsedLevelHigh:
			weights += 50
		case ModelUsedLevelSuperHigh:
			weights += 80
		}
		return int(hash[0]) < (255 * weights / 100)
	}
	return false
}

func (w *imageWorker) distributeToUnstall(models SortedModelDetailInfos, info omanager.NodeInfoResponse, ops []*omanager.ModelOperate) {

}

func (w *imageWorker) distributeToInstall(models SortedModelDetailInfos, info omanager.NodeInfoResponse, ops []*omanager.ModelOperate) {
	totalWorker := w.manager.WorkerCount()
	hash := sha3.Sum256([]byte(info.Info.MinerPubkey))

	mode := GreedyMode // 贪婪模式

	if totalWorker > 10 {
		mode = HashingMode // 散列模式
	}

	for _, model := range models {
		if w.CanInstall(model) && w.distributeMatch(mode, model, hash[:]) {
			ops = append(ops, w.getOp(model, omanager.ModelOperateType_INSTALL))
		}
	}
}

func (w *imageWorker) distributeToRun(models SortedModelDetailInfos, info omanager.NodeInfoResponse, ops []*omanager.ModelOperate) {

}

func (w *imageWorker) distributeToStopRun(models SortedModelDetailInfos, info omanager.NodeInfoResponse, ops []*omanager.ModelOperate) {

}

func (w *imageWorker) distribute() {
	models := w.modelLibrary.AllModel()
	sort.Sort(models)

	info := w.getInfo()
	operates := make([]*omanager.ModelOperate, 0)

	w.distributeToUnstall(models, info, operates)
	w.distributeToInstall(models, info, operates)
	w.distributeToRun(models, info, operates)
	w.distributeToStopRun(models, info, operates)

}
