package server

import (
	"errors"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	log "github.com/sirupsen/logrus"
	"sync"
	"time"
)

type dispatchTask struct {
	create     time.Time
	resultTime time.Time
	worker     *Worker
	task       *odysseus.TaskContent
	status     TaskStatus
	ack        chan interface{}
	errCh      chan error
	mux        sync.Mutex
	result     chan *omanager.SubmitTaskResult
}

func newDispatchTask(w *Worker, task *odysseus.TaskContent) *dispatchTask {
	return &dispatchTask{
		create: time.Now(),
		worker: w,
		task:   task,
		status: TASK_CREATE,
		ack:    make(chan interface{}, 1),
		errCh:  make(chan error, 10),
		result: make(chan *omanager.SubmitTaskResult, 1),
	}
}
func (d *dispatchTask) dispatched(wm *WorkerManager) {
	ackTicker := time.NewTicker(time.Second * 5)
	defer ackTicker.Stop()

	maxExec := d.task.TaskMaxExecTime
	if maxExec <= 0 {
		maxExec = 300 // set default to 5min.
	}
	resultTicker := time.NewTicker(time.Second * time.Duration(maxExec))
	defer resultTicker.Stop()

	d.setStatus(TASK_WAIT_ACK)
	var (
		t1     = time.Now()
		l      = log.WithField("task-id", d.task.TaskId)
		result *omanager.SubmitTaskResult
		wait   = true
	)
	for wait {
		select {
		case ackMsg := <-d.ack:
			msg := ackMsg.(*omanager.SubmitTaskAck)
			l.WithFields(log.Fields{
				"canExecute":   msg.CanExecute,
				"bootup-time":  msg.BootUpTime,
				"queue-time":   msg.QueueWaitTime,
				"execute-time": msg.ExecuteTime,
				"ackcost":      time.Since(t1).Milliseconds(),
			}).Debug("got ack message")
			if msg.CanExecute {
				d.setStatus(TASK_ACKED)
				d.errCh <- nil
				wait = true
			} else {
				// stop wait and return.
				d.errCh <- errors.New("worker can't execute task")
				wait = false
			}

		case <-ackTicker.C:
			d.setStatus(TASK_ACK_TIMEOUT)
			d.errCh <- errors.New("ack timeout")
			wait = false

		case <-resultTicker.C:
			d.setStatus(TASK_TIMEOUT)
			result = &omanager.SubmitTaskResult{
				TaskId:              d.task.TaskId,
				IsSuccessed:         false,
				TaskExecuteDuration: uint64(maxExec * 1000 * 1000), // 微秒
				TaskExecuteError:    ErrExecuteTimeout.Error(),
			}
			wait = false
		case r := <-d.result:
			d.errCh <- nil
			d.setStatus(TASK_FINISHED)
			result = r
			wait = false
		}
	}
	d.resultTime = time.Now()
	l.WithFields(log.Fields{
		"totaltime": d.resultTime.Sub(d.create).Milliseconds(),
		"status":    d.status,
	}).Debug("task finished")

	if result != nil {
		_, err := wm.taskResult(d.worker, d.task, result)
		if err != nil {
			log.WithError(err).Error("task result failed")
		}
		if d.task.TaskKind != odysseus.TaskKind_StandardTask {
			wm.Payment(d.task)
		}
	}

	return

}
func (d *dispatchTask) setAck(msg *omanager.SubmitTaskAck) {
	select {
	case d.ack <- msg:
	default:
	}
}

func (d *dispatchTask) setResult(result *omanager.SubmitTaskResult) {
	select {
	case d.result <- result:
	default:
	}
}

func (d *dispatchTask) setStatus(status TaskStatus) {
	d.mux.Lock()
	d.status = status
	d.mux.Unlock()
}
