package server

import (
	"context"
	"errors"
	"fmt"
	"github.com/gomodule/redigo/redis"
	"github.com/google/uuid"
	"github.com/odysseus/mogo/operator"
	"github.com/odysseus/mogo/types"
	"github.com/odysseus/nodemanager/config"
	log "github.com/sirupsen/logrus"
	"go.mongodb.org/mongo-driver/mongo"
	"strconv"
	"strings"
	"time"
)

func (wm *WorkerManager) updateWorkerInfo(worker *Worker, winfo *operator.WorkerInfo) error {
	// 2. update worker running info.
	wm.workerRunningOperator.DeleteByWorkerId(context.Background(), worker.WorkerAccount().String())
	runningList := make([]*operator.WorkerRunningInfo, 0)
	for _, running := range worker.info.Models.RunningModels {
		id, _ := strconv.Atoi(running.ModelId)
		iInfo := &operator.WorkerRunningInfo{
			WorkerId: worker.WorkerAccount().String(),
			ModelId:  id,
			ExecTime: int(running.ExecTime),
		}
		runningList = append(runningList, iInfo)

	}
	res, err := wm.workerRunningOperator.InsertMany(context.Background(), runningList)
	if err != nil {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
		}).WithError(err).Error("insert worker running model info failed")
	} else {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
			"count":  len(res.InsertedIDs),
		}).Debug("insert worker running model info success")
	}

	// 3. update worker installed info.
	wm.workerInstalledOperator.DeleteByWorkerId(context.Background(), worker.WorkerAccount().String())
	installList := make([]*operator.WorkerInstalledInfo, 0)
	for _, installed := range worker.info.Models.InstalledModels {
		id, _ := strconv.Atoi(installed.ModelId)
		iInfo := &operator.WorkerInstalledInfo{
			WorkerId: worker.WorkerAccount().String(),
			ModelId:  id,
		}
		if len(worker.info.Hardware.GPU) > 0 {
			iInfo.GpuFree = worker.info.Hardware.GPU[0].MemFree
		}
		installList = append(installList, iInfo)
	}
	res, err = wm.workerInstalledOperator.InsertMany(context.Background(), installList)
	if err != nil {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
		}).WithError(err).Error("insert worker installed model info failed")
	} else {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
			"count":  len(res.InsertedIDs),
		}).Debug("insert worker installed model info success")
	}

	// 1. update worker info.
	winfo.Hardware = types.PbToHardwareInfo(worker.info.Hardware)
	winfo.Models = types.PbToModelInfo(worker.info.Models)
	winfo.NodeInfo = types.PbToNodeInfo(worker.info.Info)
	err = wm.workerInfoOperator.UpdateWorker(context.Background(), winfo)
	if err != nil {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
		}).WithError(err).Error("update worker info failed")
		return err
	}

	return nil
}

func (wm *WorkerManager) addWorkerInfo(worker *Worker) error {
	// 1. add worker info.
	_, err := wm.workerInfoOperator.InsertWorker(context.Background(), &operator.WorkerInfo{
		WorkerId: worker.WorkerAccount().String(),
		NodeInfo: types.PbToNodeInfo(worker.info.Info),
		Models:   types.PbToModelInfo(worker.info.Models),
		Hardware: types.PbToHardwareInfo(worker.info.Hardware),
	})
	if err != nil {
		return err
	}

	// 2. add worker running info.
	runningList := make([]*operator.WorkerRunningInfo, 0)
	for _, running := range worker.info.Models.RunningModels {
		id, _ := strconv.Atoi(running.ModelId)
		iInfo := &operator.WorkerRunningInfo{
			WorkerId: worker.WorkerAccount().String(),
			ModelId:  id,
			ExecTime: int(running.ExecTime),
		}
		runningList = append(runningList, iInfo)

	}
	res, err := wm.workerRunningOperator.InsertMany(context.Background(), runningList)
	if err != nil {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
		}).WithError(err).Error("insert worker running model info failed")
	} else {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
			"count":  len(res.InsertedIDs),
		}).Debug("insert worker running model info success")
	}

	// 3. add worker installed info.
	installList := make([]*operator.WorkerInstalledInfo, 0)
	for _, installed := range worker.info.Models.InstalledModels {
		id, _ := strconv.Atoi(installed.ModelId)
		iInfo := &operator.WorkerInstalledInfo{
			WorkerId: worker.WorkerAccount().String(),
			ModelId:  id,
		}
		if len(worker.info.Hardware.GPU) > 0 {
			iInfo.GpuFree = worker.info.Hardware.GPU[0].MemFree
		}
		installList = append(installList, iInfo)

	}
	res, err = wm.workerInstalledOperator.InsertMany(context.Background(), installList)
	if err != nil {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
		}).WithError(err).Error("insert worker installed model info failed")
	} else {
		log.WithFields(log.Fields{
			"worker": worker.WorkerAccount().String(),
			"count":  len(res.InsertedIDs),
		}).Debug("insert worker installed model info success")
	}
	return nil
}

func (wm *WorkerManager) AddWorker(worker *Worker) error {
	// 1. if worker is exist in mogo, update worker info.
	winfo, err := wm.workerInfoOperator.FindWorkerByWorkerId(context.Background(), worker.WorkerAccount().String())
	if err != nil {
		if err == mongo.ErrNoDocuments {
			// create a new
			return wm.addWorkerInfo(worker)
		} else {
			log.WithError(err).Error("find worker info failed")
		}
	} else {
		if winfo != nil {
			// update worker info.
			return wm.updateWorkerInfo(worker, winfo)
		}
	}
	return errors.New("can't replace worker info")
}

func (wm *WorkerManager) LastNmValue(worker *Worker) string {
	nmlist, _ := wm.WorkerNmList(worker)
	if len(nmlist) == 0 {
		return ""
	}
	for _, nmvalue := range nmlist {
		endpoint, _ := wm.parseWorkerNmValue(nmvalue)
		if endpoint == config.GetConfig().PublicEndpoint() {
			return nmvalue
		}
	}
	return ""
}

func (wm *WorkerManager) deleteOldNmValue(worker *Worker) {
	nmlist, _ := wm.WorkerNmList(worker)
	for _, nmvalue := range nmlist {
		endpoint, _ := wm.parseWorkerNmValue(nmvalue)
		if endpoint == config.GetConfig().PublicEndpoint() {
			wm.rdb.SRem(context.Background(), workerStatusKey(worker), nmvalue)
		}
	}
}

func (wm *WorkerManager) addNewNmValue(worker *Worker) error {
	split := "#"
	v := fmt.Sprintf("%s%s%d", config.GetConfig().PublicEndpoint(), split, time.Now().Unix())
	worker.latestNmValue = v
	return wm.rdb.SAdd(context.Background(), workerStatusKey(worker), v).Err()
}

func (wm *WorkerManager) UpdateWorkerActive(worker *Worker) {
	if !worker.online {
		return
	}
	wm.deleteOldNmValue(worker)
	err := wm.addNewNmValue(worker)
	if err != nil {
		log.WithError(err).Error("add new nm value failed")
	}
}

func (wm *WorkerManager) parseWorkerNmValue(nmValue string) (string, int64) {
	split := "#"
	strs := strings.Split(nmValue, split)
	if len(strs) == 2 {
		endpoint := strs[0]
		timestamp, _ := strconv.ParseInt(strs[1], 10, 64)
		return endpoint, timestamp
	}
	return "", 0
}

func (wm *WorkerManager) WorkerNmList(worker *Worker) ([]string, error) {
	return wm.rdb.SMembers(context.Background(), workerStatusKey(worker)).Result()
}

func (wm *WorkerManager) InActiveWorker(worker *Worker) {
	wm.deleteOldNmValue(worker)
	nmlist, err := wm.WorkerNmList(worker)
	if err == nil && len(nmlist) == 0 {
		wm.rdb.Del(context.Background(), workerStatusKey(worker))
		if worker.info != nil {
			wm.delWorkerFromWhiteListSet(worker, worker.info.Info.BenefitAddress)
			wm.rdb.Del(context.Background(), workerLastTaskTmKey(worker))
			// delete worker info from mogo.
			n, err := wm.workerInfoOperator.DeleteByWorkerId(context.Background(), worker.WorkerAccount().String())
			if err != nil {
				log.WithError(err).Error("delete worker info failed")
			}
			log.Debugf("delete worker info count %d", n)
			n, err = wm.workerRunningOperator.DeleteByWorkerId(context.Background(), worker.WorkerAccount().String())
			log.Debugf("delete worker running info count %d", n)
			n, err = wm.workerInstalledOperator.DeleteByWorkerId(context.Background(), worker.WorkerAccount().String())
			log.Debugf("delete worker installed info count %d", n)
		}
	}
}

func (wm *WorkerManager) addWorkerToWhiteListSet(worker *Worker, benefit string) {
	wm.rdb.SAdd(context.Background(), workerSetsKey(benefit), worker.workerAddr)
}

func (wm *WorkerManager) delWorkerFromWhiteListSet(worker *Worker, benefit string) {
	wm.rdb.SRem(context.Background(), workerSetsKey(benefit), worker.workerAddr)
}

func (wm *WorkerManager) getWorkerWhiteListSetByBenefit(benefit string) ([]string, error) {
	list, err := wm.rdb.SMembers(context.Background(), workerSetsKey(benefit)).Result()
	if err == redis.ErrNil {
		return []string{}, nil
	}
	return list, err
}

func (wm *WorkerManager) getWorkerLastTaskTime(worker *Worker) (int64, error) {
	return wm.rdb.Get(context.Background(), workerLastTaskTmKey(worker)).Int64()
}

func (wm *WorkerManager) setWorkerLastTaskTime(worker *Worker, tm int64) error {
	return wm.rdb.Set(context.Background(), workerLastTaskTmKey(worker), tm, 0).Err()
}

func (wm *WorkerManager) checkWhiteList(worker *Worker, benefit string) error {
	return nil
	wh, err := wm.node.Cache().GetWhWithAddr(benefit)
	if err != nil {
		log.WithFields(log.Fields{
			"worker":  worker.uuid,
			"benefit": benefit,
		}).Error("worker not in white list")
		return errors.New("not in white list")
	}
	maxNodeCount := wh.NodeNum
	nodeList, err := wm.getWorkerWhiteListSetByBenefit(benefit)
	if err != nil {
		return errors.New("check worker white list failed")
	}
	nodeInSet := false
	for _, node := range nodeList {
		if strings.Compare(strings.ToLower(worker.workerAddr), strings.ToLower(node)) == 0 {
			nodeInSet = true
			break
		}
	}
	if nodeInSet || len(nodeList) < int(maxNodeCount) {
		return nil
	} else {
		return errors.New("reach max nodenum with the benefit")
	}
}

func workerSetsKey(benefit string) string {
	return config.WORKER_SETS_PREFIX + benefit
}

func workerStatusKey(w *Worker) string {
	id := workerId(w)
	return fmt.Sprintf("%s_%s", config.WORKER_STATUS_PREFIX, id)
}

func workerLastTaskTmKey(w *Worker) string {
	return config.WORKER_LAST_TASK_TM_PREFIX + w.workerAddr
}

func workerUid(w *Worker) string {
	id := workerId(w)
	uuid := uuid.NewString()
	return fmt.Sprintf("%s:%s", id, uuid[:4])
}

func workerId(w *Worker) string {
	return fmt.Sprintf("%s_%d", w.workerAddr, w.nonce)
}
