package utils

import (
	"crypto/ecdsa"
	"encoding/hex"
	"github.com/ethereum/go-ethereum/crypto"
	"strings"
)

func HexToPrivatekey(key string) (*ecdsa.PrivateKey, error) {
	return crypto.HexToECDSA(key)
}

func PrivatekeyToHex(key *ecdsa.PrivateKey) string {
	return hex.EncodeToString(crypto.FromECDSA(key))
}

func PrivatekeyToAddress(key *ecdsa.PrivateKey) string {
	return crypto.PubkeyToAddress(key.PublicKey).String()
}

func PubkeyToAddress(key *ecdsa.PublicKey) string {
	return crypto.PubkeyToAddress(*key).String()
}

func PubkeyToHex(key *ecdsa.PublicKey) string {
	pub := crypto.FromECDSAPub(key)
	return hex.EncodeToString(pub)
}

func HexToPubkey(key string) (*ecdsa.PublicKey, error) {
	if strings.HasPrefix(key, "0x") {
		key = key[2:]
	}
	pub, err := hex.DecodeString(key)
	if err != nil {
		return nil, err
	}
	if len(pub) > 65 {
		pub = pub[:65]
	}
	return crypto.UnmarshalPubkey(pub)
}

func VerifySignature(data []byte, signature []byte, oripubkey []byte) bool {
	dataHash := crypto.Keccak256Hash(data)
	if len(signature) == 65 {
		signature = signature[:64]
	}
	pubLen := len(oripubkey)
	pubkey := []byte{}

	if pubLen == 130 || pubLen == 132 {
		pubkey = FromHex(string(oripubkey))
	} else if pubLen == 65 {
		pubkey = oripubkey
	}
	verified := crypto.VerifySignature(pubkey, dataHash[:], signature)

	return verified
}
