package nmregister

import (
	"crypto/ecdsa"
	"encoding/json"
	"fmt"
	"github.com/odysseus/nodemanager/config"
	"github.com/odysseus/nodemanager/utils"
	"github.com/odysseus/service-registry/common"
	"github.com/odysseus/service-registry/query"
	"github.com/odysseus/service-registry/registry"
	"github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"os"
	"sync"
	"time"
)

// nmregistry implement register nodemanger to redis, and get nodemanger list from redis.
// filter some nodemanager to worker.

type FullNodeManagerInfo struct {
	Reginfo registry.RegistryInfo
	NmInfo  query.NodeManagerInfo
}

type NodeManagerInfo struct {
	Pubkey   string `json:"pubkey"`
	Endpoint string `json:"endpoint"`
}

type NMRegister struct {
	querier  *query.ServiceQuerier
	nodelist []FullNodeManagerInfo
	rdb      *redis.Client
	conf     *config.Config
	rw       sync.RWMutex
	public   ecdsa.PublicKey
	quit     chan struct{}
}

func (s *NMRegister) ServiceType() common.ServiceType {
	return common.SERVICE_NODE_MANAGER
}

func (s *NMRegister) Instance() string {
	hname, _ := os.Hostname()
	return fmt.Sprintf("%s:%s", s.conf.RemoteHost, hname)
}

func (s *NMRegister) Status() string {
	return "running"
}

func (s *NMRegister) DetailInfo() (json.RawMessage, error) {
	priv, err := utils.HexToPrivatekey(s.conf.PrivateKey)
	if err != nil {
		panic(fmt.Sprintf("invalid private key: %s", err))
	}

	addr := utils.PrivatekeyToHex(priv)
	pubHex := utils.PubkeyToHex(&priv.PublicKey)

	endpoint := s.conf.PublicEndpoint()

	info := query.NodeManagerInfo{}
	info.Addr = addr
	info.Pubkey = pubHex
	info.Endpoint = endpoint
	return json.Marshal(info)
}

func NewNMRegister(conf *config.Config, querier *query.ServiceQuerier, public ecdsa.PublicKey) *NMRegister {
	return &NMRegister{
		querier: querier,
		conf:    conf,
		public:  public,
		quit:    make(chan struct{}),
	}
}

func (s *NMRegister) Start() {
	refresh := time.NewTicker(time.Second * 5)
	defer refresh.Stop()

	for {
		select {
		case <-s.quit:
			return
		case <-refresh.C:
			if nodes, err := s.allNodeManager(); err != nil {
				log.WithError(err).Error("refresh all nodemanager failed")
			} else {
				s.rw.Lock()
				s.nodelist = nodes
				s.rw.Unlock()
				refresh.Reset(time.Second * 60)
			}

		}
	}
}

func (s *NMRegister) Stop() {
	close(s.quit)
}

func (s *NMRegister) GetNodeManagerList(filter ManagerFilter) []NodeManagerInfo {
	s.rw.RLock()
	defer s.rw.RUnlock()
	var ret []NodeManagerInfo

	for _, node := range s.nodelist {
		if filter == nil || filter.Filter(node) {
			ret = append(ret, NodeManagerInfo{
				Pubkey:   node.NmInfo.Pubkey,
				Endpoint: node.NmInfo.Endpoint,
			})
		}
	}
	return ret
}

func (s *NMRegister) allNodeManager() ([]FullNodeManagerInfo, error) {
	var ret []FullNodeManagerInfo
	var tsExpired = 100
	nmlist, err := s.querier.Select(common.SERVICE_NODE_MANAGER).List()
	if err != nil {
		return nil, err
	}
	for _, v := range nmlist {
		var info registry.RegistryInfo
		var nminfo query.NodeManagerInfo
		if err := json.Unmarshal([]byte(v), &info); err != nil {
			continue
		}
		if err := json.Unmarshal([]byte(info.Detail), &nminfo); err != nil {
			continue
		}
		if time.Now().Unix()-info.Timestamp > int64(tsExpired) {
			// heart beat expired, ignore this nodemanager
			continue
		}
		ret = append(ret, FullNodeManagerInfo{
			Reginfo: info,
			NmInfo:  nminfo,
		})
	}
	return ret, nil
}
