package server

import (
	"crypto/ecdsa"
	"errors"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/odysseus/nodemanager/utils"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	log "github.com/sirupsen/logrus"
	"math/big"
	"time"
)

func (wm *WorkerManager) taskResult(worker *Worker, task *odysseus.TaskContent, result *omanager.SubmitTaskResult) (*omanager.ManagerMessage_ProofTaskResult, error) {
	switch task.TaskKind {
	case odysseus.TaskKind_ComputeTask:
		return wm.computeTaskResult(worker, task, result)
	case odysseus.TaskKind_StandardTask:
		return wm.standardTaskResult(worker, task, result)
	}

	return nil, errors.New("unsupport task kind")
}

func (wm *WorkerManager) computeTaskResult(worker *Worker, task *odysseus.TaskContent, result *omanager.SubmitTaskResult) (*omanager.ManagerMessage_ProofTaskResult, error) {
	if worker.info.nodeInfo == nil {
		return nil, errors.New("unknown worker node info")
	}
	log.WithFields(log.Fields{
		"task-id":               task.TaskId,
		"task-type":             task.TaskType,
		"task-kind":             task.TaskKind,
		"task-success":          result.IsSuccessed,
		"task-execute-duration": result.TaskExecuteDuration,
	}).Debug("got task result")

	if result.IsSuccessed == false {
		taskResponse := &odysseus.TaskResponse{
			TaskId:           task.TaskId,
			TaskResultHeader: result.TaskResultHeader,
			TaskResultBody:   result.TaskResultBody,
			TaskUid:          task.TaskUid,
			TaskFee:          task.TaskFee,
			TaskIsSucceed:    false,
			TaskError:        "worker failed",
		}
		receipt := wm.makeReceipt(worker, task, result, errors.New("worker failed"))
		wm.node.PostResult(receipt)
		go wm.doCallback(task.TaskCallback, taskResponse)
		return nil, nil
	} else {
		taskResponse := &odysseus.TaskResponse{
			TaskId:           task.TaskId,
			TaskResultHeader: result.TaskResultHeader,
			TaskResultBody:   result.TaskResultBody,
			TaskUid:          task.TaskUid,
			TaskFee:          task.TaskFee,
			TaskIsSucceed:    true,
			TaskError:        "",
		}
		go wm.doCallback(task.TaskCallback, taskResponse)
	}

	//{
	//	// verify container_signature and miner_signature
	//	// container_signature = sign(hash(task_id+hash(task_param)+hash(task_result)))
	//	paramHash := crypto.Keccak256Hash(task.TaskParam)
	//	resultHash := crypto.Keccak256Hash(result.TaskResult)
	//	dataHash := crypto.Keccak256Hash(utils.CombineBytes([]byte(result.TaskUuid), paramHash[:], resultHash[:]))
	//	containerPubkey, _ := utils.HexToPubkey(hex.EncodeToString(task.ContainerPubkey))
	//	verified := ecdsa.VerifyASN1(containerPubkey, dataHash[:], result.ContainerSignature)
	//	if !verified {
	//		// todo: handle signature verify failed
	//	}
	//}
	{
		// verify miner_signature
		// miner_signature = sign(hash((task_id+hash(task_param)+hash(task_result)))
		paramHash := crypto.Keccak256Hash(task.TaskParam)
		resultHash := crypto.Keccak256Hash(result.TaskResultBody)
		dataHash := crypto.Keccak256Hash(utils.CombineBytes([]byte(result.TaskId), paramHash[:], resultHash[:]))
		minerPubkey, _ := utils.HexToPubkey(worker.info.nodeInfo.MinerPubkey)
		verified := ecdsa.VerifyASN1(minerPubkey, dataHash[:], result.MinerSignature)
		log.WithField("minerSignatureVerify", verified).Debug("miner signature verify")
		if !verified {
			// todo: handle signature verify failed
		}
	}

	receipt := wm.makeReceipt(worker, task, result, Succeed)
	wm.node.PostResult(receipt)

	//manager_signature = sign(hash((task_id+hash(task_param)+hash(task_result)+container_signature+miner_signature+workload+time))
	now := time.Now().Unix()
	paramHash := crypto.Keccak256Hash(task.TaskParam)
	resultHash := crypto.Keccak256Hash(result.TaskResultBody)
	dataHash := crypto.Keccak256Hash(utils.CombineBytes([]byte(result.TaskId), paramHash[:], resultHash[:],
		worker.ProfitAccount().Bytes(), worker.WorkerAccount().Bytes(), result.ContainerSignature, result.MinerSignature, big.NewInt(int64(task.TaskWorkload)).Bytes()),
		big.NewInt(now).Bytes())

	signature, err := wm.node.Sign(dataHash[:])
	if err != nil {
		log.WithError(err).Error("sign result failed")
		return nil, err
	}

	proof := new(omanager.ManagerMessage_ProofTaskResult)
	proof.ProofTaskResult = &omanager.ProofTaskResult{
		TaskId:           result.TaskId,
		ManagerSignature: signature,
		Timestamp:        uint64(now),
		Workload:         uint64(task.TaskWorkload),
		ContainerPubkey:  utils.CombineBytes(task.ContainerPubkey),
	}
	log.WithFields(log.Fields{
		"task-id":  result.TaskId,
		"workload": task.TaskWorkload,
	}).Debug("send proof to worker")
	return proof, nil
}

func (wm *WorkerManager) standardTaskResult(worker *Worker, task *odysseus.TaskContent, result *omanager.SubmitTaskResult) (*omanager.ManagerMessage_ProofTaskResult, error) {
	if worker.info.nodeInfo == nil {
		return nil, errors.New("unknown worker node info")
	}
	log.WithFields(log.Fields{
		"task-id":               task.TaskId,
		"task-type":             task.TaskType,
		"task-kind":             task.TaskKind,
		"task-success":          result.IsSuccessed,
		"task-execute-duration": result.TaskExecuteDuration,
	}).Debug("got task result")

	if result.IsSuccessed == false {
		receipt := wm.makeReceipt(worker, task, result, errors.New("worker failed"))
		wm.node.PostResult(receipt)
		return nil, nil
	}

	stdlib := wm.std.GetStdLib(task.TaskType)
	if stdlib == nil {
		log.WithField("task-id", task.TaskId).Error("not found stdlib to verify")
		return nil, errors.New("not found stdlib to verify")
	}

	if stdlib.VerifyResult(string(task.TaskParam), result.TaskResultBody) == true {
		log.WithField("task-id", task.TaskId).Debug("stdlib to verify passed")
	} else {
		log.WithField("task-id", task.TaskId).Debug("stdlib to verify failed")
		return nil, errors.New("stdlib to verify failed")
	}

	//{
	//	// verify container_signature and miner_signature
	//	// container_signature = sign(hash(task_id+hash(task_param)+hash(task_result)))
	//	paramHash := crypto.Keccak256Hash(task.TaskParam)
	//	resultHash := crypto.Keccak256Hash(result.TaskResult)
	//	dataHash := crypto.Keccak256Hash(utils.CombineBytes([]byte(result.TaskUuid), paramHash[:], resultHash[:]))
	//	containerPubkey, _ := utils.HexToPubkey(hex.EncodeToString(task.ContainerPubkey))
	//	verified := ecdsa.VerifyASN1(containerPubkey, dataHash[:], result.ContainerSignature)
	//	if !verified {
	//		// todo: handle signature verify failed
	//	}
	//}
	{
		// verify miner_signature
		// miner_signature = sign(hash((task_id+hash(task_param)+hash(task_result)))
		paramHash := crypto.Keccak256Hash(task.TaskParam)
		resultHash := crypto.Keccak256Hash(result.TaskResultBody)
		dataHash := crypto.Keccak256Hash(utils.CombineBytes([]byte(result.TaskId), paramHash[:], resultHash[:]))
		minerPubkey, _ := utils.HexToPubkey(worker.info.nodeInfo.MinerPubkey)
		verified := ecdsa.VerifyASN1(minerPubkey, dataHash[:], result.MinerSignature)
		log.WithField("minerSignatureVerify", verified).Debug("miner signature verify")
		if !verified {
			// todo: handle signature verify failed
		}
	}

	receipt := wm.makeReceipt(worker, task, result, Succeed)
	wm.node.PostResult(receipt)

	now := time.Now().Unix()
	//manager_signature = sign(hash((task_id+hash(task_param)+hash(task_result)+container_signature+miner_signature+workload+time))
	paramHash := crypto.Keccak256Hash(task.TaskParam)
	resultHash := crypto.Keccak256Hash(result.TaskResultBody)
	dataHash := crypto.Keccak256Hash(utils.CombineBytes([]byte(result.TaskId), paramHash[:], resultHash[:],
		worker.ProfitAccount().Bytes(), worker.WorkerAccount().Bytes(), result.ContainerSignature, result.MinerSignature, big.NewInt(int64(task.TaskWorkload)).Bytes()),
		big.NewInt(now).Bytes())

	signature, err := wm.node.Sign(dataHash[:])
	if err != nil {
		log.WithError(err).Error("sign result failed")
		return nil, err
	}

	proof := new(omanager.ManagerMessage_ProofTaskResult)
	proof.ProofTaskResult = &omanager.ProofTaskResult{
		TaskId:           result.TaskId,
		ManagerSignature: signature,
		Timestamp:        uint64(now),
		Workload:         uint64(task.TaskWorkload),
		ContainerPubkey:  utils.CombineBytes(task.ContainerPubkey),
	}
	log.WithFields(log.Fields{
		"task-id":  result.TaskId,
		"workload": task.TaskWorkload,
	}).Debug("send proof to worker")
	return proof, nil
}
