package main

import (
	"context"
	"flag"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	log "github.com/sirupsen/logrus"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
)

var (
	manager = flag.String("manager", "127.0.0.1:10001", "seed manager endpoint")
)

func newManagerClient(endpoint string) (omanager.NodeManagerServiceClient, error) {
	client, err := grpc.Dial(endpoint,
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithDefaultCallOptions(
			grpc.MaxCallRecvMsgSize(1024*1024*1024),
			grpc.MaxCallSendMsgSize(1024*1024*1024)),
	)
	if err != nil {
		return nil, err
	}
	return omanager.NewNodeManagerServiceClient(client), nil
}

func withManager(idx int, endpoint string) {
	client, err := newManagerClient(endpoint)
	if err != nil {
		log.WithField("endpoint", endpoint).Fatal("connect manager failed", "err", err)
		return
	}
	managerClient, err := client.RegisterWorker(context.Background())
	if err != nil {
		log.WithField("endpoint", endpoint).Fatal("register worker failed", "err", err)
		return
	}
	minerPubkey := "0x04d4c8fdd1729112459ba4e4a3c66e1cf480dfc1b06e250880234dc0578cb765280114711dd73c86ddec0e264d5a49b49558827d67d0dcdac6207bdfb1264c7461"
	if idx == 0 {

		reg := &omanager.WorkerMessage{
			Message: &omanager.WorkerMessage_RegisteMessage{
				RegisteMessage: &omanager.RegisteMessage{
					DeviceIp:    "192.168.1.112",
					MinerPubkey: minerPubkey,
				},
			},
		}
		managerClient.Send(reg)
		dinfo := &omanager.DeviceInfo{
			DeviceType:  "gpu-0",
			DeviceModel: "rtx-3080",
			DeviceParam: "device param",
			DevicePower: 100,
		}

		devInfo := &omanager.WorkerMessage{
			Message: &omanager.WorkerMessage_DeviceInfo{
				DeviceInfo: &omanager.DeviceInfoMessage{
					Devices:         []*omanager.DeviceInfo{dinfo},
					DeviceSignature: make([]byte, 0),
				},
			},
		}
		managerClient.Send(devInfo)

		resource := &omanager.WorkerMessage{
			Message: &omanager.WorkerMessage_ResourceMap{
				ResourceMap: &omanager.SubmitResourceMap{
					ResourceMap: make([]byte, 10),
				},
			},
		}
		managerClient.Send(resource)
	}
	taskCh := make(chan *omanager.ManagerMessage_PushTaskMessage, 1000)
	go func() {
		for {
			select {
			case task, ok := <-taskCh:
				if !ok {
					return
				}
				demoResult := "a demo task result return"
				msg := &omanager.WorkerMessage{
					Message: &omanager.WorkerMessage_SubmitTaskResult{
						SubmitTaskResult: &omanager.SubmitTaskResult{
							TaskId:              task.PushTaskMessage.TaskId,
							ContainerSignature:  make([]byte, 65),
							MinerSignature:      make([]byte, 65),
							TaskResultBody:      []byte(demoResult),
							IsSuccessed:         true,
							TaskExecuteDuration: 10000,
						},
					},
				}
				if err := managerClient.Send(msg); err != nil {
					log.WithField("endpoint", endpoint).Fatal("response device info failed", "err", err)
				}

			}
		}
	}()
	for {
		data, err := managerClient.Recv()
		if err != nil {
			log.WithField("endpoint", endpoint).Error("register worker failed", "err", err)
			return
		}
		switch b := data.Message.(type) {
		case *omanager.ManagerMessage_HeartbeatRequest:
			log.WithField("endpoint", endpoint).Info("got heartbeat")
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_HeartbeatResponse{
					HeartbeatResponse: &omanager.HeartbeatResponse{
						Timestamp: b.HeartbeatRequest.Timestamp,
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response heartbeat failed", "err", err)
				return
			}
		case *omanager.ManagerMessage_NodeInfoRequest:
			log.WithField("endpoint", endpoint).Info("got node info request")
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_NodeInfo{
					NodeInfo: &omanager.NodeInfoResponse{
						MinerPubkey:    minerPubkey,
						BenefitAddress: "0x0Fb196385c8826e3806ebA2cA2cb78B26E08fEEe",
						DeviceIp:       "192.168.1.112",
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response heartbeat failed", "err", err)
				return
			}

		case *omanager.ManagerMessage_DeviceUsage:
			log.WithField("endpoint", endpoint).Info("got device usage")
			usage := &omanager.DeviceUsage{
				DeviceType:  "gpu",
				DeviceUsage: 120,
			}
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_DeviceUsage{
					DeviceUsage: &omanager.DeviceUsageResponse{
						Usage: []*omanager.DeviceUsage{
							usage,
						},
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response device info failed", "err", err)
				return
			}
		case *omanager.ManagerMessage_StatusRequest:
			log.WithField("endpoint", endpoint).Info("got status request")
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_Status{
					Status: &omanager.StatusResponse{
						DeviceStatus: []byte{0x1},
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response device info failed", "err", err)
				return
			}

		case *omanager.ManagerMessage_GoodbyeMessage:
			log.WithField("endpoint", endpoint).Info("got goodbye message")
			return
		case *omanager.ManagerMessage_PushTaskMessage:
			log.WithField("endpoint", endpoint).Info("got push task message")
			taskCh <- b

		case *omanager.ManagerMessage_ProofTaskResult:
			log.WithField("endpoint", endpoint).Info("got proof task result")

		}
	}
}

func main() {
	flag.Parse()
	seed, err := newManagerClient(*manager)
	if err != nil {
		log.Fatal("connect seed manager failed", "err", err)
	}
	nmlist, err := seed.ManagerList(context.Background(), new(omanager.ManagerListRequest))
	if err != nil {
		log.Fatal("get manager list failed", "err", err)
	}
	log.WithField("count", len(nmlist.Managers)).Info("got manager list")
	if len(nmlist.Managers) == 0 {
		log.Fatal("no manager found")
	}
	for i, nm := range nmlist.Managers {
		endpoint := nm.Endpoint
		go withManager(i, endpoint)
	}
	var wait = make(chan struct{})
	<-wait
}
