package server

import (
	"context"
	"crypto/ecdsa"
	"github.com/IBM/sarama"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/odysseus/cache/cachedata"
	"github.com/odysseus/cache/model"
	"github.com/odysseus/nodemanager/config"
	"github.com/odysseus/nodemanager/nmregister"
	"github.com/odysseus/nodemanager/utils"
	basev1 "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"github.com/odysseus/service-registry/query"
	"github.com/odysseus/service-registry/registry"
	"github.com/redis/go-redis/v9"
	log "github.com/sirupsen/logrus"
	"google.golang.org/grpc"
	"net"
	"strings"
)

type Node struct {
	register      *nmregister.NMRegister
	registry      *registry.Registry
	apiServer     *grpc.Server
	rdb           *redis.Client
	wm            *WorkerManager
	privk         *ecdsa.PrivateKey
	cache         *cachedata.CacheData
	kafkaProducer sarama.AsyncProducer
	taskResultCh  chan *basev1.TaskReceipt
	taskProofCh   chan *basev1.TaskProof
}

func NewNode() *Node {
	redisConfig := config.GetConfig().Redis
	privk, err := utils.HexToPrivatekey(config.GetConfig().PrivateKey)
	if err != nil {
		log.WithError(err).Error("failed to parse node manager private key")
		return nil
	}

	querier := query.NewQuery(registry.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	})

	register := nmregister.NewNMRegister(config.GetConfig(), querier, privk.PublicKey)

	reg := registry.NewRegistry(registry.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	}, register)

	rdb := utils.NewRedisClient(utils.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	})
	dbconf := config.GetConfig().DbConfig
	pay := cachedata.NewCacheData(context.Background(), cachedata.RedisConnParam{
		Addr:     redisConfig.Addr,
		Password: redisConfig.Password,
		DbIndex:  redisConfig.DbIndex,
	}, model.DbConfig{
		Host:   dbconf.Host,
		Port:   dbconf.Port,
		User:   dbconf.User,
		Passwd: dbconf.Passwd,
		DbName: dbconf.DbName,
	})

	brokers := strings.Split(config.GetConfig().Kafka.Brokers, ";")
	producer, _ := utils.NewKafkaProducer(brokers)
	node := &Node{
		register:      register,
		registry:      reg,
		rdb:           rdb,
		privk:         privk,
		cache:         pay,
		apiServer:     grpc.NewServer(grpc.MaxSendMsgSize(1024*1024*20), grpc.MaxRecvMsgSize(1024*1024*20)),
		kafkaProducer: producer,
		taskResultCh:  make(chan *basev1.TaskReceipt, 100000),
		taskProofCh:   make(chan *basev1.TaskProof, 100000),
	}

	node.wm = NewWorkerManager(rdb, node)
	log.WithField("nm pubkey", utils.PubkeyToHex(&privk.PublicKey)).Info("node manager started")

	return node
}

func (n *Node) Sign(hash []byte) ([]byte, error) {
	return crypto.Sign(hash, n.privk)
}

func (n *Node) Start() error {
	go n.registry.Start()
	go n.register.Start()
	go n.postLoop()

	if err := n.apiStart(); err != nil {
		return err
	}
	return nil
}

func (n *Node) apiStart() error {
	lis, err := net.Listen("tcp", config.GetConfig().ApiEndpoint())
	if err != nil {
		log.WithError(err).Error("failed to listen endpoint")
		return err
	}

	omanager.RegisterNodeManagerServiceServer(n.apiServer, &NodeManagerService{
		quit: make(chan struct{}),
		node: n,
	})

	err = n.apiServer.Serve(lis)
	if err != nil {
		log.WithError(err).Error("failed to serve apiserver")
		return err
	}
	log.Info("api started")
	return nil
}

func (n *Node) PostResult(result *basev1.TaskReceipt) {
	defer func() {
		// handler recover
		if err := recover(); err != nil {
			log.WithError(err.(error)).Error("post result panic")
		}
	}()
	n.taskResultCh <- result
}

func (n *Node) PostProof(proof *basev1.TaskProof) {
	defer func() {
		// handler recover
		if err := recover(); err != nil {
			log.WithError(err.(error)).Error("post result panic")
		}
	}()
	n.taskProofCh <- proof
}

func (n *Node) postLoop() {
	for {
		select {
		case receipt, ok := <-n.taskResultCh:
			if !ok {
				return
			}
			if err := utils.FireTaskReceipt(n.kafkaProducer, receipt, config.GetConfig().Kafka.ReceiptTopic); err != nil {
				log.WithError(err).Error("fire task receipt to kafka failed")
			}
		case proof, ok := <-n.taskProofCh:
			if !ok {
				return
			}
			if err := utils.FireTaskProof(n.kafkaProducer, proof, config.GetConfig().Kafka.ProofTopic); err != nil {
				log.WithError(err).Error("fire task receipt to kafka failed")
			}
		}
	}
}

func (n *Node) Stop() {
	n.registry.Stop()
	n.register.Stop()
	n.apiServer.Stop()
	close(n.taskResultCh)
	close(n.taskProofCh)
}
