package server

import (
	"context"
	"encoding/hex"
	"errors"
	"fmt"
	"github.com/gomodule/redigo/redis"
	"github.com/odysseus/nodemanager/config"
	log "github.com/sirupsen/logrus"
	"strconv"
	"strings"
	"time"
)

func (wm *WorkerManager) UpdateWorkerDeviceStatusInfo(worker *Worker, status []byte) {
	wm.rdb.Set(context.Background(), workerDeviceStatusInfoKey(worker), status, 0)
}

func (wm *WorkerManager) UpdateWorkerUsageInfo(worker *Worker, usageInfo string) {
	wm.rdb.Set(context.Background(), workerUsageInfoKey(worker), usageInfo, 0)
}

func (wm *WorkerManager) UpdateWorkerDeviceInfo(worker *Worker, deviceInfos string) {
	wm.rdb.Set(context.Background(), workerDeviceInfoKey(worker), deviceInfos, 0)
}

func (wm *WorkerManager) UpdateWorkerResourceInfo(worker *Worker, resourceInfo []byte) {
	rstr := hex.EncodeToString(resourceInfo)
	wm.rdb.Set(context.Background(), workerResourceInfoKey(worker), rstr, 0)
}

func (wm *WorkerManager) UpdateWorkerNonce(worker *Worker, nonce int) error {
	return wm.rdb.Set(context.Background(), workerNonceKey(worker), nonce, 0).Err()
}

func (wm *WorkerManager) GetWorkerNonce(worker *Worker) (int, error) {
	if worker.workerAddr != "" {
		nonceK := workerNonceKey(worker)
		nonce, err := wm.rdb.Get(context.Background(), nonceK).Int()
		if err == redis.ErrNil {
			nonce = 1
			if err = wm.rdb.Set(context.Background(), nonceK, nonce, 0).Err(); err != nil {
				return 0, err
			}
		}
		return nonce, nil
	}
	return 0, errors.New("unkown worker node info")
}

func (wm *WorkerManager) IncrWorkerNonce(worker *Worker) (int, error) {
	nonce, err := wm.rdb.Incr(context.Background(), workerNonceKey(worker)).Uint64()
	return int(nonce), err
}

func (wm *WorkerManager) AddWorkerFirst(worker *Worker) error {
	log.WithField("worker", worker.workerAddr).Info("add worker first time.")
	wm.UpdateWorkerActive(worker)
	for _, device := range worker.info.deviceInfo.Devices {
		if !strings.HasPrefix(device.DeviceType, "gpu") {
			continue
		}
		// add device to redis
		priority := 0
		_ = device // todo: set priority with device info.
		// add worker to redis queue
		if err := wm.rdb.RPush(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(priority), workerId(worker)).Err(); err != nil {
			continue
		}
	}

	return nil
}

func (wm *WorkerManager) AddWorkerToQueue(worker *Worker) {
	nonce, err := wm.GetWorkerNonce(worker)
	if err != nil {
		log.WithField("worker-addr", worker.workerAddr).Error("get worker nonce failed when get device info")
	} else {
		// if statekeys not exist, nonce don't change.
		nmlist, err := wm.WorkerNmList(worker)
		if err != nil {
			if err == redis.ErrNil {
				wm.UpdateWorkerActive(worker)
			}
		} else {
			if len(nmlist) == 0 {
				// if nmlist is empty, nonce incr.
				nonce, err = wm.IncrWorkerNonce(worker)
				if err != nil {
					log.WithField("worker-addr", worker.workerAddr).Error("incr worker nonce failed when get device info")
				}
			} else {
				// else if nmlist is not empty, clear and add self to it.
				wm.rdb.Del(context.Background(), workerStatusKey(worker))
				wm.UpdateWorkerActive(worker)
			}
		}
	}
	if err == nil {
		worker.nonce = nonce
		wm.AddWorkerFirst(worker)
		worker.addFirstSucceed = true
	}
}

func (wm *WorkerManager) AddWorkerSingle(worker *Worker) error {
	log.WithField("worker", worker.workerAddr).Info("add worker on back.")
	wm.UpdateWorkerActive(worker)
	{
		// add worker to redis queue
		priority := 0
		if err := wm.rdb.RPush(context.Background(), config.WORKER_QUEUE_PREFIX+strconv.Itoa(priority), workerId(worker)).Err(); err != nil {
			log.WithError(err).Error("add worker back to queue failed.")
		}
	}
	// add worker to redis queue
	return nil
}

func (wm *WorkerManager) UpdateWorkerActive(worker *Worker) {
	if !worker.online {
		return
	}
	nonce, err := wm.GetWorkerNonce(worker)
	if err != nil {
		return
	}
	if nonce != worker.nonce {
		wm.InActiveWorker(worker)
		worker.nonce = nonce
	}

	old := worker.latestNmValue
	if err := wm.activeWorker(worker); err != nil {
		return
	}
	wm.rdb.SRem(context.Background(), workerStatusKey(worker), old)
}

func (wm *WorkerManager) activeWorker(worker *Worker) error {
	split := "#"
	v := fmt.Sprintf("%s%s%d", config.GetConfig().PublicEndpoint(), split, time.Now().Unix())
	worker.latestNmValue = v
	return wm.rdb.SAdd(context.Background(), workerStatusKey(worker), v).Err()
}

func (wm *WorkerManager) parseWorkerNmValue(nmValue string) (string, int64) {
	split := "#"
	strs := strings.Split(nmValue, split)
	if len(strs) == 2 {
		endpoint := strs[0]
		timestamp, _ := strconv.ParseInt(strs[1], 10, 64)
		return endpoint, timestamp
	}
	return "", 0
}

func (wm *WorkerManager) WorkerNmList(worker *Worker) ([]string, error) {
	return wm.rdb.SMembers(context.Background(), workerStatusKey(worker)).Result()
}

func (wm *WorkerManager) InActiveWorker(worker *Worker) {
	wm.rdb.SRem(context.Background(), workerStatusKey(worker), worker.latestNmValue)

	if list, err := wm.rdb.SMembers(context.Background(), workerStatusKey(worker)).Result(); err == nil && len(list) == 0 {
		wm.rdb.Del(context.Background(), workerStatusKey(worker))
		wm.rdb.Del(context.Background(), workerUsageInfoKey(worker))
		wm.rdb.Del(context.Background(), workerDeviceInfoKey(worker))
		wm.rdb.Del(context.Background(), workerResourceInfoKey(worker))
	}
}

func workerResourceInfoKey(w *Worker) string {
	return config.WORKER_RESOURCE_INFO_PREFIX + w.workerAddr
}

func workerDeviceInfoKey(w *Worker) string {
	return config.WORKER_DEVICE_INFO_PREFIX + w.workerAddr
}

func workerUsageInfoKey(w *Worker) string {
	return config.WORKER_USAGE_INFO_PREFIX + w.workerAddr
}

func workerDeviceStatusInfoKey(w *Worker) string {
	return config.WORKER_DEVICE_STATUS_PREFIX + w.workerAddr
}

func workerNonceKey(w *Worker) string {
	return config.WORKER_NONCE_KEY_PREFIX + w.workerAddr
}

func workerStatusKey(w *Worker) string {
	id := workerId(w)
	return fmt.Sprintf("%s_%s", config.WORKER_STATUS_PREFIX, id)
}

func workerId(w *Worker) string {
	return fmt.Sprintf("%s_%d", w.workerAddr, w.nonce)
}
