package main

import (
	"context"
	"crypto/ecdsa"
	"crypto/rand"
	"flag"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/odysseus/nodemanager/utils"
	omanager "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	log "github.com/sirupsen/logrus"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
)

var (
	manager = flag.String("manager", "127.0.0.1:10001", "seed manager endpoint")
)

func newManagerClient(endpoint string) (omanager.NodeManagerServiceClient, error) {
	client, err := grpc.Dial(endpoint,
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithDefaultCallOptions(
			grpc.MaxCallRecvMsgSize(1024*1024*1024),
			grpc.MaxCallSendMsgSize(1024*1024*1024)),
	)
	if err != nil {
		return nil, err
	}
	return omanager.NewNodeManagerServiceClient(client), nil
}

func main() {
	flag.Parse()
	seed, err := newManagerClient(*manager)
	if err != nil {
		log.Fatal("connect seed manager failed", "err", err)
	}
	nmlist, err := seed.ManagerList(context.Background(), new(omanager.ManagerListRequest))
	if err != nil {
		log.Fatal("get manager list failed", "err", err)
	}
	log.WithField("count", len(nmlist.Managers)).Info("got manager list")
	if len(nmlist.Managers) == 0 {
		log.Fatal("no manager found")
	}
	endpoint := nmlist.Managers[0].Endpoint

	client, err := newManagerClient(endpoint)
	if err != nil {
		log.WithField("endpoint", endpoint).Fatal("connect manager failed", "err", err)
		return
	}
	managerClient, err := client.RegisterWorker(context.Background())
	if err != nil {
		log.WithField("endpoint", endpoint).Fatal("register worker failed", "err", err)
		return
	}
	sk, _ := ecdsa.GenerateKey(crypto.S256(), rand.Reader)

	for {
		data, err := managerClient.Recv()
		if err != nil {
			log.WithField("endpoint", endpoint).Fatal("register worker failed", "err", err)
			return
		}
		switch b := data.Message.(type) {
		case *omanager.ManagerMessage_HeartbeatRequest:
			log.WithField("endpoint", endpoint).Info("got heartbeat")
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_HeartbeatResponse{
					HeartbeatResponse: &omanager.HeartbeatResponse{
						Timestamp: b.HeartbeatRequest.Timestamp,
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response heartbeat failed", "err", err)
				return
			}
		case *omanager.ManagerMessage_DeviceRequest:
			log.WithField("endpoint", endpoint).Info("got device request")
			deviceInfo := &omanager.DeviceInfo{
				DeviceType:  "gpu",
				DeviceModel: "nvidia",
				DeviceParam: "12",
				DevicePower: 1000,
			}
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_DeviceInfo{
					DeviceInfo: &omanager.DeviceInfoResponse{
						MinerPubkey: utils.PubkeyToHex(&sk.PublicKey),
						Devices:     []*omanager.DeviceInfo{deviceInfo},
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response device info failed", "err", err)
				return
			}

		case *omanager.ManagerMessage_DeviceUsage:
			log.WithField("endpoint", endpoint).Info("got device usage")
			usage := &omanager.DeviceUsage{
				DeviceType:  "gpu",
				DeviceUsage: 120,
			}
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_DeviceUsage{
					DeviceUsage: &omanager.DeviceUsageResponse{
						Usage: []*omanager.DeviceUsage{
							usage,
						},
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response device info failed", "err", err)
				return
			}
		case *omanager.ManagerMessage_StatusRequest:
			log.WithField("endpoint", endpoint).Info("got status request")
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_Status{
					Status: &omanager.StatusResponse{
						DeviceStatus: []byte{0x1},
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response device info failed", "err", err)
				return
			}

		case *omanager.ManagerMessage_GoodbyeMessage:
			log.WithField("endpoint", endpoint).Info("got goodbye message")
			return
		case *omanager.ManagerMessage_PushTaskMessage:
			log.WithField("endpoint", endpoint).Info("got push task message")
			demoResult := "a demo task result return"
			msg := &omanager.WorkerMessage{
				Message: &omanager.WorkerMessage_SubmitTaskResult{
					SubmitTaskResult: &omanager.SubmitTaskResult{
						TaskUuid:           b.PushTaskMessage.TaskUuid,
						ContainerSignature: make([]byte, 65),
						MinerSignature:     make([]byte, 65),
						TaskResult:         []byte(demoResult),
					},
				},
			}
			if err := managerClient.Send(msg); err != nil {
				log.WithField("endpoint", endpoint).Fatal("response device info failed", "err", err)
				return
			}

		case *omanager.ManagerMessage_ProofTaskResult:
			log.WithField("endpoint", endpoint).Info("got proof task result")

		}

	}
}
