package standardlib

import (
	"encoding/json"
	"github.com/ethereum/go-ethereum/common"
	"github.com/odysseus/nodemanager/config"
	odysseus "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	log "github.com/sirupsen/logrus"
	"os"
	"sync"
)

type StdLib interface {
	GenerateParam(difficult int) (string, error)
	VerifyResult(param string, result []byte) bool
}

type StandardTasks struct {
	tasks    []StdTask
	mux      sync.Mutex
	stdTasks sync.Map
}

var (
	standardLibMap = sync.Map{}
)

func RegisterStdLib(name string, lib StdLib) {
	standardLibMap.Store(name, lib)
}

func NewStandardTasks() *StandardTasks {
	s := &StandardTasks{}
	s.loadTasks()
	return s
}

func (s *StandardTasks) loadTasks() {
	s.tasks = loadStdTasks()
	for _, task := range s.tasks {
		s.stdTasks.Store(task.TaskType, task)
	}
}

func (s *StandardTasks) GetTask(taskType uint64) (StdTask, bool) {
	if task, ok := s.stdTasks.Load(taskType); ok {
		return task.(StdTask), true
	}
	return StdTask{}, false
}

func (s *StandardTasks) GetStdLib(taskType uint64) StdLib {
	if task, exist := s.GetTask(taskType); exist {
		return getStdLib(task.Standlib)
	}
	return nil
}

func getStdLib(name string) StdLib {
	if lib, ok := standardLibMap.Load(name); ok {
		return lib.(StdLib)
	}
	return nil
}

type StdTask struct {
	odysseus.TaskContent
	Standlib string
}

type defineStdTask struct {
	TaskKind        int    `json:"task_kind,omitempty"`
	TaskType        uint64 `json:"task_type,omitempty"`
	TaskCmd         string `json:"task_cmd,omitempty"`
	TaskParam       string `json:"task_param,omitempty"`
	TaskFee         string `json:"task_fee,omitempty"`
	TaskWorkload    int64  `json:"task_workload,omitempty"`
	ContainerPubkey string ` json:"container_pubkey,omitempty"`
	Standlib        string `json:"standlib,omitempty"`
}

func loadStdTasks() []StdTask {
	file := config.GetConfig().StandardTaskFilePath()
	defStdTasks := make([]defineStdTask, 0)
	// read content from file and json unmarshal to defStdTasks
	data, err := os.ReadFile(file)
	if err != nil {
		log.WithError(err).Error("read standard task file failed")
		return []StdTask{}
	}
	err = json.Unmarshal(data, &defStdTasks)
	if err != nil {
		log.WithError(err).Error("json unmarshal standard task file failed")
		return []StdTask{}
	}
	stdTasks := make([]StdTask, len(defStdTasks))
	for i, defStdTask := range defStdTasks {
		stdTasks[i].TaskKind = odysseus.TaskKind_StandardTask
		stdTasks[i].TaskType = defStdTask.TaskType
		stdTasks[i].TaskCmd = defStdTask.TaskCmd
		stdTasks[i].TaskParam = []byte(defStdTask.TaskParam)
		stdTasks[i].TaskFee = defStdTask.TaskFee
		stdTasks[i].TaskWorkload = defStdTask.TaskWorkload
		stdTasks[i].ContainerPubkey = common.Hex2Bytes(defStdTask.ContainerPubkey)
		stdTasks[i].Standlib = defStdTask.Standlib
	}

	return stdTasks
}
