package nm

import (
	"bytes"
	"encoding/base64"
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/golang/groupcache/lru"
	baseV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/base/v1"
	nodeManagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"io"
	"math/rand"
	"mime/multipart"
	"net/http"
	"strconv"
	"strings"
	"sync"
	"time"
)

type TaskHandler struct {
	Wg                 *sync.WaitGroup
	Mutex              *sync.Mutex
	LruCache           *lru.Cache
	DockerOp           *operate.DockerOp
	CmdOp              *operate.Command
	TaskMsg            chan *nodeManagerV1.PushTaskMessage
	HttpClient         *http.Client
	IsExecAiTask       bool
	IsExecStandardTask bool
}

var oldTaskImageName string

func NewTaskWorker(op *operate.DockerOp) *TaskHandler {
	return &TaskHandler{
		Wg:           &sync.WaitGroup{},
		Mutex:        &sync.Mutex{},
		LruCache:     lru.New(100),
		DockerOp:     op,
		TaskMsg:      make(chan *nodeManagerV1.PushTaskMessage, 0),
		HttpClient:   &http.Client{},
		IsExecAiTask: false,
	}
}

func (t *TaskHandler) HandlerTask(runCount int) {
	for i := 0; i < runCount; i++ {
		go func(t *TaskHandler) {
			for {
				select {
				case taskMsg := <-t.TaskMsg:
					{
						switch taskMsg.TaskKind {
						case baseV1.TaskKind_SystemTask:
							{
								//command := operate.GetSystemCommand(taskMsg.TaskCmd, taskMsg.TaskParam, taskMsg.TaskId+".sh")
								t.SystemTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_ComputeTask:
							{
								t.ComputeTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_CustomTask:
							{
								t.CustomTaskHandler(taskMsg)
							}
						case baseV1.TaskKind_StandardTask:
							{
								t.ComputeTaskHandler(taskMsg)
							}
						}
					}
				}
			}
		}(t)
	}
}

func (t *TaskHandler) SystemTaskHandler(taskMsg *nodeManagerV1.PushTaskMessage) {
	defer t.Wg.Done()

	log.Info("received systemTask--------------------------------")
}

func (t *TaskHandler) ComputeTaskHandler(taskMsg *nodeManagerV1.PushTaskMessage) {
	defer t.Wg.Done()
	taskExecResult := &models.TaskResult{
		TaskHttpStatusCode: 200,
		TaskRespBody:       nil,
		TaskHttpHeaders:    nil,
		TaskIsSuccess:      false,
		TaskExecTime:       0,
	}
	taskCmd := &models.TaskCmd{}
	err := json.Unmarshal(bytes.NewBufferString(taskMsg.TaskCmd).Bytes(), taskCmd)
	if err != nil {
		log.Errorf("failed to unmarshal task cmd: %s", err.Error())
		return
	}
	log.Info("received task cmd :", taskCmd)

	if taskMsg.TaskKind == baseV1.TaskKind_ComputeTask {
		t.IsExecAiTask = true
		if t.IsExecStandardTask {
			//todo: 停止标准任务容器
			//containers := t.DockerOp.ListContainer()
			//for _, container := range containers {
			//	if container.Image == taskCmd.ImageName && container.State == "running" {
			//		t.DockerOp.StopContainer(container.ID)
			//	}
			//}
			t.IsExecStandardTask = false
		}
	} else if taskMsg.TaskKind == baseV1.TaskKind_StandardTask {
		t.IsExecStandardTask = true
	}
	if oldTaskImageName != "" && oldTaskImageName != taskCmd.ImageName {
		//todo: 停止标准任务容器
		//containers := t.DockerOp.ListContainer()
		//for _, container := range containers {
		//	if container.Image == taskCmd.ImageName && container.State == "running" {
		//		t.DockerOp.StopContainer(container.ID)
		//	}
		//}
		oldTaskImageName = taskCmd.ImageName
	}
	images, err := t.DockerOp.PsImages()
	if err != nil {
		log.Error("Ps images failed:", err)
		return
	}
	imageId := ""
	isFound := false
	for _, image := range images {
		if isFound {
			break
		}
		for _, tag := range image.RepoTags {
			if tag == taskCmd.ImageName {
				imageId = image.ID
				isFound = true
				log.Info("The image found success:", image.ID)
				break
			}
		}
	}
	if !isFound {
		log.Error("The image is not found:", taskCmd.ImageName)
		return
	}
	running, internalIp, internalPort := t.foundImageIsRunning(imageId)
	if !running {
		var externalPort int64
		for {
			// 设置种子以确保每次运行时生成不同的随机数序列
			rand.Seed(time.Now().UnixNano())
			// 生成一个介于 0 和 100 之间的随机整数
			externalPort = rand.Int63n(10001) + 10000
			log.Info("DockerOp UsedExternalPort :", t.DockerOp.UsedExternalPort[externalPort])
			if t.DockerOp.UsedExternalPort[externalPort] {
				continue
			}
			break
		}
		taskCmd.DockerCmd.HostIp = "0.0.0.0"
		taskCmd.DockerCmd.HostPort = strconv.FormatInt(externalPort, 10)
		//if int64(len(containers)) == conf.GetConfig().ContainerNum {
		//	//todo: 待定，需要根据权重去停止哪个容器
		//	t.DockerOp.StopAndDeleteContainer(containers[0].ID)
		//}
		containerId, err := t.DockerOp.CreateAndStartContainer(taskCmd.ImageName, taskCmd.DockerCmd)
		if err != nil {
			log.Errorf("Create and start container failed: %s", err.Error())
			return
		}
		log.Infof("Started container with ID %s", containerId)
		time.Sleep(time.Second * 20)
		running, internalIp, internalPort = t.foundImageIsRunning(imageId)
		if running {
			taskCmd.ApiUrl = fmt.Sprintf("http://%s:%d%s", internalIp, internalPort, taskCmd.ApiUrl)
			log.Info("Container ports:", internalPort)
			log.WithField("ApiUrl", taskCmd.ApiUrl).Info("The image is not  running")
		}
	} else {
		taskCmd.ApiUrl = fmt.Sprintf("http://%s:%d%s", internalIp, internalPort, taskCmd.ApiUrl)
		log.Info("Container ports:", internalPort)
		log.WithField("ApiUrl", taskCmd.ApiUrl).Info("The image is running")
	}
	startBeforeTaskTime := time.Now()
	taskParam := &models.TaskParam{}
	err = json.Unmarshal(taskMsg.TaskParam, taskParam)
	if err != nil {
		log.WithField("err", err).Error("Error unmarshalling task parameter")
		return
	}
	reqContainerBody := bytes.NewReader(taskParam.Body)
	if len(taskParam.Queries) > 0 {
		queryString := utils.MatchContainerQueryString(taskParam.Queries)
		taskCmd.ApiUrl = fmt.Sprintf("%s?%s", taskCmd.ApiUrl, queryString)
	}
	post, err := t.HttpClient.Post(taskCmd.ApiUrl, "application/json", reqContainerBody)
	if err != nil {
		log.WithField("error:", err).Error("Http client post request container failed")
		return
	}
	endAfterTaskTime := time.Since(startBeforeTaskTime)
	log.WithField("time", endAfterTaskTime.Seconds()).WithField("taskId", taskMsg.TaskId).Info("Exec task end (second is units) :")
	log.WithField("StatusCode", post.StatusCode).WithField("taskId", taskMsg.TaskId).Info("Exec task result")
	if post.StatusCode == http.StatusOK {
		taskExecResult.TaskHttpStatusCode = http.StatusOK
		readBody, err := io.ReadAll(post.Body)
		if err != nil {
			log.Error("received error: ", err)
			return
		}
		isUseFileCache := true
		if taskMsg.TaskKind != baseV1.TaskKind_StandardTask {
			for key, value := range taskParam.Headers {
				if key == models.UseFileCache {
					if value[0] == "0" {
						isUseFileCache = false
						break
					}
				}
			}
			log.WithField("isUseFileCache", isUseFileCache).Info("is use file cache")
			if isUseFileCache && readBody != nil {
				containerResp := &models.ModelResponse{}
				//log.WithField("task resp", readBody).Info("received response")
				err = json.Unmarshal(readBody, &containerResp)
				if err != nil {
					log.WithError(err).Error("Error unmarshalling oss resp body failed")
					return
				}
				if len(containerResp.Output) == 1 {
					if utils.IsBase64ImageStr(containerResp.Output[0]) {
						containerRespOutput := strings.SplitN(containerResp.Output[0], ",", 2)
						imageStr := containerRespOutput[1]
						queryString := utils.MatchFileCacheQueryString(taskParam.Headers, taskCmd.ImageName, t.DockerOp.ModelsInfo, containerRespOutput[0])
						//ossUri, err := t.uploadOSS(taskMsg.TaskId, queryString, containerResp.Output[0])
						ossUri, err := t.uploadOSS(taskMsg.TaskId, queryString, imageStr)
						if err != nil {
							log.WithError(err).Error("upload image into file cache failed")
							return
						}
						log.WithField("uri", ossUri).Info("upload image OSS  successful")
						if ossUri != "" {
							taskExecResult.TaskHttpStatusCode = models.RedirectCode
							post.Header.Set("Location", ossUri)
						}
					}
				}
			}
		}
		headers, err := json.Marshal(post.Header)
		if err != nil {
			log.Error("JSON marshal header error: ", err)
			return
		}
		log.WithField("headers", post.Header).Info("return task http headers")
		if taskMsg.TaskKind == baseV1.TaskKind_StandardTask && !isUseFileCache {
			taskExecResult.TaskRespBody = readBody
		}
		taskExecResult.TaskHttpHeaders = headers
		taskExecResult.TaskIsSuccess = true
		taskExecResult.TaskExecTime = endAfterTaskTime.Microseconds()
	} else {
		taskExecResult.TaskHttpStatusCode = int32(post.StatusCode)
		if post.Body != nil {
			all, err := io.ReadAll(post.Body)
			if err != nil {
				log.Error("JSON read error: ", err)
				return
			}
			taskExecResult.TaskRespBody = all
		} else {
			taskExecResult.TaskRespBody = nil
		}
		log.WithField("error", post.Body).WithField("taskId", taskMsg.TaskId).Error("Exec task result is failed")
	}
	if taskMsg.TaskKind == baseV1.TaskKind_ComputeTask {
		t.IsExecAiTask = false
	} else if taskMsg.TaskKind == baseV1.TaskKind_StandardTask {
		t.IsExecStandardTask = false
	}
	t.Mutex.Lock()
	t.LruCache.Add(taskMsg.TaskId, taskExecResult)
	t.Mutex.Unlock()
	//log.WithField("result", taskExecResult).Info("lru cache storage task result")
	log.Info("received computeTask--------------------------------")
}

func (t *TaskHandler) CustomTaskHandler(taskMsg *nodeManagerV1.PushTaskMessage) {
	defer t.Wg.Done()
	_, err := t.DockerOp.PsImages()
	if err != nil {
		log.Error("custom task handler docker op ps images failed: ", err)
		return
	}
	log.Info("received customTask--------------------------------")
}

func (t *TaskHandler) GetMinerSign(msg *nodeManagerV1.PushTaskMessage, taskResult []byte) ([]byte, []byte, []byte) {
	reqHash := crypto.Keccak256Hash(msg.TaskParam)
	respHash := crypto.Keccak256Hash(taskResult)
	signHash := crypto.Keccak256Hash(bytes.NewBufferString(msg.TaskId).Bytes(), reqHash.Bytes(), respHash.Bytes())
	sign, err := crypto.Sign(signHash.Bytes(), conf.GetConfig().SignPrivateKey)
	if err != nil {
		log.Error("custom task handler")
	}
	log.Info("Miner sign:", common.Bytes2Hex(sign))
	return reqHash.Bytes(), respHash.Bytes(), sign
}

func (t *TaskHandler) foundImageIsRunning(imageId string) (bool, string, uint16) {
	containers := t.DockerOp.ListContainer()
	for _, container := range containers {
		if container.ImageID == imageId && container.State == "running" {
			networks := container.NetworkSettings.Networks
			ip := ""
			for _, endPoint := range networks {
				ip = endPoint.IPAddress
				log.Warn("Container network ip:", ip)
			}
			return true, ip, container.Ports[0].PrivatePort
		}
	}
	return false, "", 0
}

func (t *TaskHandler) uploadOSS(taskId string, queries string, base64Image string) (string, error) {
	// todo: 解析结果
	// TODO: 存储OSS
	var requestBody bytes.Buffer
	writer := multipart.NewWriter(&requestBody)
	// 创建文件表单字段
	fileField, err := writer.CreateFormFile("file", fmt.Sprintf("%s.png", taskId))
	if err != nil {
		log.WithError(err).Error("Error creating form file")
		return "", err
	}
	// 将 base64 解码后的内容复制到表单字段
	decodedImage, err := base64.StdEncoding.DecodeString(base64Image)
	if err != nil {
		log.WithError(err).Error("Error decoding base64 image")
		return "", err
	}
	_, err = io.Copy(fileField, bytes.NewReader(decodedImage))
	//_, err = io.Copy(fileField, strings.NewReader(base64Image))
	if err != nil {
		log.WithError(err).Error("Error copying file contents")
		return "", err
	}
	// 关闭 multipart writer
	err = writer.Close()
	if err != nil {
		log.WithError(err).Error("Error closing writer")
		return "", err
	}
	ossUrl := fmt.Sprintf("%s?%s", conf.GetConfig().OssUrl, queries)
	request, err := http.NewRequest("POST", ossUrl, &requestBody)
	if err != nil {
		return "", err
	}
	request.Header.Set("Content-Type", writer.FormDataContentType())
	response, err := t.HttpClient.Do(request)
	if err != nil {
		log.WithError(err).Error("Error request oss failed")
		return "", err
	}
	ossRespBody, err := io.ReadAll(response.Body)
	if err != nil {
		log.WithError(err).Error("Error read oss resp body failed")
		return "", err
	}
	return bytes.NewBuffer(ossRespBody).String(), nil
}
