package conf

import (
	"crypto/ecdsa"
	"encoding/json"
	"example.com/m/log"
	"example.com/m/utils"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/spf13/viper"
	"io/ioutil"
)

type Config struct {
	NmSeed              string `json:"nm_seed"`
	SignPrv             string
	SignPrivateKey      *ecdsa.PrivateKey
	SignPub             string
	SignPublicAddress   common.Address
	DockerServer        string `json:"docker_server"`
	HeartRespTimeSecond int64  `json:"heart_response"`
	HeartRespTimeMillis int64
	TaskValidatorTime   float64 `json:"task_validator_time"`
	BenefitAddress      string  `json:"benefit_address"`
	ContainerNum        int64   `json:"container_num"`
	NodeManagerNum      int64   `json:"node_manager_num"`
	ChainID             int64   `json:"chain_id"`
}

var _cfg *Config = nil

func init() {
	// 设置配置文件的名称（不包含扩展名)
	viper.SetConfigName("config")

	// 设置配置文件的类型
	viper.SetConfigType("json")

	// 设置配置文件所在的目录
	viper.AddConfigPath("./")

	// 读取配置文件
	if err := viper.ReadInConfig(); err != nil {
		fmt.Println("Error reading config file:", err)
		return
	}

	configFilePath := viper.ConfigFileUsed()
	if configFilePath == "" {
		// handle error
		log.Error("config file path is empty")
		panic("config file path is empty")
	}

	data, err := ioutil.ReadFile(configFilePath)
	if err != nil {
		// handle error
		log.Error("Read cfg file error:", err)
		panic("Read cfg file error")
	}
	err = json.Unmarshal(data, &_cfg)
	if err != nil {
		// handle error
		log.Error("Json unmarshal cfg error:", err)
		panic("Json unmarshal cfg error")
	}
	_cfg.HeartRespTimeMillis = _cfg.HeartRespTimeSecond * 60 * 60 * 1000
	prvKey, err := utils.GetPrv()
	if err != nil {
		panic("get prv error or delete keystore after restart")
	}
	_cfg.SignPrivateKey = prvKey
	ecdsaPub := prvKey.PublicKey
	_cfg.SignPub = common.Bytes2Hex(crypto.FromECDSAPub(&ecdsaPub))
	log.Info("PublicKey", _cfg.SignPub)
	publicAddr := crypto.PubkeyToAddress(ecdsaPub)
	log.Info("publicAddr:", publicAddr)
	_cfg.SignPublicAddress = publicAddr
}

func GetConfig() *Config {
	return _cfg
}

func (c *Config) SetRewardAddress(addr string) bool {
	isAddr := common.IsHexAddress(addr)
	if isAddr {
		c.BenefitAddress = addr
	}
	return isAddr
}

func (c *Config) SetNmSeed(seed string) {
	c.NmSeed = seed
}
