package conf

import (
	"crypto/ecdsa"
	"example.com/m/log"
	"fmt"
	"github.com/ethereum/go-ethereum/common"
	"net/url"
)

type Config struct {
	SignPrv                  string
	SignPub                  string
	DockerServer             string
	BenefitAddress           string
	HeartRespTimeMillis      int64
	ExternalIp               string
	SignPublicAddress        common.Address
	SignPrivateKey           *ecdsa.PrivateKey
	NmSeed                   string  `json:"nm_seed"`
	HeartRespTimeSecond      int64   `json:"heart_response"`
	TaskValidatorTime        float64 `json:"task_validator_time"`
	ContainerNum             int64   `json:"container_num"`
	NodeManagerNum           int64   `json:"node_manager_num"`
	ChainID                  int64   `json:"chain_id"`
	ApiUrl                   string  `json:"api_url"`
	ValidatorUrl             string  `json:"validator_url"`
	OssUrl                   string  `json:"oss_url"`
	WaitLastTaskExecTime     int64   `json:"wait_last_task_exec_time"`
	OpSys                    string  `json:"op_sys"`
	ReplicateImageNameSuffix string  `json:"replicate_image_name_suffix"`
	IsStopLastContainer      bool    `json:"is_stop_last_container"`
}

var _cfg *Config = nil

func init() {
	_cfg = &Config{}
}

func GetConfig() *Config {
	return _cfg
}

func (c *Config) SetExternalIp(externalIp string) {
	log.Info("Setting external ip to:", externalIp)
	c.ExternalIp = externalIp
}

func (c *Config) GetExternalIp() string {
	return c.ExternalIp
}

func (c *Config) SetRewardAddress(addr string) bool {
	isAddr := common.IsHexAddress(addr)
	if isAddr {
		c.BenefitAddress = addr
	}
	return isAddr
}

func (c *Config) SetDockerServerUrl(url string) bool {
	isSuccess, newUrl := checkDockerServer(url)
	log.Info("Setting Docker Server URL:", newUrl)
	if isSuccess {
		c.DockerServer = newUrl
	}
	return isSuccess
}

func (c *Config) SetNmSeed(seed string) {
	c.NmSeed = seed
}

func (c *Config) SetOpSys(sys string) bool {
	if sys != "win" && sys != "mac" && sys != "linux" {
		return false
	}
	c.OpSys = sys
	return true
}

func checkDockerServer(rawURL string) (bool, string) {
	if rawURL == "" {
		return true, fmt.Sprintf("tcp://%s:%s", "host.docker.internal", "2375")
	}
	parsedURL, err := url.Parse(rawURL)
	if err != nil {
		log.Error("Error parsing URL:", err)
		return false, ""
	}
	// 提取协议、主机和端口信息
	protocol := parsedURL.Scheme
	if parsedURL.Scheme != "" && protocol != "tcp" {
		protocol = "tcp://"
	}
	ip := "host.docker.internal"
	port := "2375"
	if parsedURL.Port() != "" {
		port = parsedURL.Port()
	}
	if parsedURL.Hostname() != "" {
		ip = parsedURL.Hostname()
	}
	newUrl := fmt.Sprintf("%s://%s:%s", protocol, ip, port)
	return true, newUrl
}
