package utils

import (
	"bytes"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"encoding/base64"
	"encoding/json"
	"example.com/m/log"
	"example.com/m/models"
	"fmt"
	"github.com/ethereum/go-ethereum/accounts/keystore"
	"github.com/ethereum/go-ethereum/crypto"
	"math/big"
	"net/http"
	"net/url"
	"os"
	"strings"
)

const KeypadPwd = "keystore"

func GenerateRandomNumber(privateKey *ecdsa.PrivateKey, length int64) *big.Int {
	// 生成一个随机数
	randNum, err := rand.Int(rand.Reader, elliptic.P256().Params().N)
	if err != nil {
		log.Error("Error generating random number:", err)
	}
	randNum.Mod(randNum, big.NewInt(length))
	log.Info("Generating random number:", randNum)
	return randNum
}

func GetPrv() (*ecdsa.PrivateKey, error) {
	if _, err := os.Stat(KeypadPwd); os.IsNotExist(err) {
		//log.Info("Keystore not found. Generating a new one...")
		// 生成私钥
		privateKey, err := generatePrivateKey()
		if err != nil {
			log.Error("Error generating private key:", err)
			return nil, err
		}
		// 保存私钥到 keystore 文件
		err = savePrivateKey(privateKey)
		if err != nil {
			log.Error("Error saving private key:", err)
			return nil, err
		}
		//log.Info("Keystore generated successfully.")
		return privateKey, nil
	} else {
		//log.Info("Keystore found. Reading private key...")
		// 读取私钥
		privateKey, err := readPrivateKey()
		if err != nil || privateKey == nil {
			log.Error("Error reading private key:", err)
			return nil, err
		}
		//log.Info("Private key read successfully:", privateKey)
		return privateKey, nil
	}
}

func generatePrivateKey() (*ecdsa.PrivateKey, error) {
	return crypto.GenerateKey()
}

func savePrivateKey(privateKey *ecdsa.PrivateKey) error {
	ks := keystore.NewKeyStore(KeypadPwd, keystore.StandardScryptN, keystore.StandardScryptP)
	account, err := ks.ImportECDSA(privateKey, KeypadPwd)
	if err != nil {
		log.Fatal(err)
	}
	fmt.Println(account.Address.Hex())
	return nil
}

func readPrivateKey() (*ecdsa.PrivateKey, error) {
	file, err := os.ReadDir(KeypadPwd)
	if err != nil {
		return nil, err
	}
	for _, info := range file {
		keystoreFile := fmt.Sprintf("%s%s%s", KeypadPwd, "/", info.Name())
		jsonBytes, err := os.ReadFile(keystoreFile)
		if err != nil {
			log.Error("import ecdsa keystore error: ", err)
			continue
		}
		key, err := keystore.DecryptKey(jsonBytes, KeypadPwd)
		if err != nil {
			log.Error("keystore decrypt key failed:", err)
			continue
		}
		return key.PrivateKey, nil
	}
	return nil, nil
}

// IsBase64ImageStr 检查字符串是否是 Base64 编码的图像数据
func IsBase64ImageStr(imageStr string) (bool, []byte, string, string) {
	// 移除可能的前缀（如 "data:image/png;base64,"）
	imageStrArr := strings.Split(imageStr, ",")
	if len(imageStrArr) < 2 {
		return false, nil, "", ""
	}
	base64CodePrefix := imageStrArr[0]
	base64Code := imageStrArr[1]
	decodeBytes, err := base64.StdEncoding.DecodeString(base64Code)
	if err != nil {
		log.WithError(err).Error("base64 decode string failed")
		return false, nil, "", ""
	}
	dataSuffix := strings.Split(base64CodePrefix, ";")
	if len(dataSuffix) < 2 {
		return false, nil, "", ""
	}
	log.Info("dataSuffix:", dataSuffix)
	formatStrArr := strings.Split(dataSuffix[0], ":")
	if len(formatStrArr) < 2 {
		return false, nil, "", ""
	}
	formatStr := formatStrArr[1]
	log.Info("formatStr:", formatStr)
	suffixArr := strings.Split(formatStr, "/")
	if len(suffixArr) < 2 {
		return false, nil, "", ""
	}
	suffix := suffixArr[1]
	return true, decodeBytes, formatStr, suffix
}

func MatchFileCacheQueryString(params map[string][]string, taskImageName string, modelsInfo []*models.ModelInfo, contentType string) string {
	values := url.Values{}
	isExistFileExpires := false
	for key, value := range params {
		if key == models.ResultFileExpiresDB {
			values.Add(key, value[0])
			isExistFileExpires = true
			break
		}
	}
	isModelExistFileExpires := false
	if !isExistFileExpires {
		for _, info := range modelsInfo {
			if info.ImageName == taskImageName && info.FileExpiresTime != "" {
				values.Add(models.ResultFileExpiresDB, info.FileExpiresTime)
				isModelExistFileExpires = true
				break
			}
		}
	}
	if !isModelExistFileExpires {
		values.Add(models.ResultFileExpiresDB, "600")
	}
	values.Add(models.ContentType, contentType)
	return values.Encode()
}

func MatchContainerQueryString(params map[string]string) string {
	values := url.Values{}
	for key, value := range params {
		values.Add(key, value)
	}
	return values.Encode()
}

func EncodeJsonEscapeHTML(apiRes any) []byte {
	apiResBody := bytes.NewBuffer([]byte{})
	encoder := json.NewEncoder(apiResBody)
	encoder.SetEscapeHTML(false)
	err := encoder.Encode(apiRes)
	if err != nil {
		log.WithError(err).Error("encoder Encode")
		return apiResBody.Bytes()
	}
	//log.WithField("apiResBody", string(apiResBody.Bytes())).Info("model resp")
	return apiResBody.Bytes()
}

func BuildParams(params ...interface{}) []interface{} {
	res := make([]interface{}, len(params))
	for i, param := range params {
		res[i] = param
	}
	return res
}

func CloneRequest(r *http.Request) *http.Request {
	copiedRequest := &http.Request{
		Method:           r.Method,
		URL:              r.URL,
		Proto:            r.Proto,
		ProtoMajor:       r.ProtoMajor,
		ProtoMinor:       r.ProtoMinor,
		Header:           make(http.Header),
		Body:             r.Body,
		ContentLength:    r.ContentLength,
		TransferEncoding: append([]string(nil), r.TransferEncoding...),
		Close:            r.Close,
		Host:             r.Host,
		RemoteAddr:       r.RemoteAddr,
		RequestURI:       r.RequestURI,
		TLS:              r.TLS,
	}
	for k, v := range r.Header {
		copiedRequest.Header[k] = v
	}
	return copiedRequest
}
