package largeModel

import (
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/db"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"fmt"
	nodemanagerV2 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"io"
	"net/http"
	"strconv"
	"strings"
	"time"
)

type ModelHandler struct {
	dockerOp *operate.DockerOp
	client   *http.Client
	IsInit   bool
}

func NewModelHandler(dockerOp *operate.DockerOp) *ModelHandler {
	return &ModelHandler{
		dockerOp: dockerOp,
		client:   &http.Client{},
		IsInit:   false,
	}
}

func (m *ModelHandler) MonitorModelInfo() {
	ticker := time.NewTicker(time.Millisecond)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			modelResp, err := m.client.Get(conf.GetConfig().ApiUrl)
			if err != nil {
				log.Error("Error getting model info from client failed:", err)
				continue
			}
			bodyBytes, err := io.ReadAll(modelResp.Body)
			if err != nil {
				log.Error("Error reading model response failed:", err)
				continue
			}
			resp := &models.Resp{}
			err = json.Unmarshal(bodyBytes, resp)
			if err != nil {
				log.Error("Unmarshal model response failed:", err)
				continue
			}
			if resp.Code != http.StatusOK {
				log.Error("Response code :", resp.Code)
				continue
			}
			if resp.Data == nil || len(resp.Data) == 0 {
				log.Warn("Response data is empty")
				continue
			}
			//imageMap, err := m.dockerOp.PsImageNameMap()
			//if err != nil {
			//	log.Error("Error getting image name map from client failed:", err)
			//	continue
			//}
			modelInfosResp := resp.Data
			for _, modelInfo := range modelInfosResp {
				if modelInfo.ImageName == "" {
					continue
				}
				modelInfo.ImageName = fmt.Sprintf("%s-%s", modelInfo.ImageName, conf.GetConfig().OpSys)
				split := strings.Split(modelInfo.ImageName, ":")
				if len(split) != 2 {
					continue
				}
				log.WithField("name", modelInfo.ImageName).Info("The image name is")
				m.dockerOp.SignApi[modelInfo.ImageName] = modelInfo.SignUrl
				model, _ := db.GetModel(modelInfo.ImageName)
				if model != nil {
					model.UpdateFiled(modelInfo)
					err := db.PutModel(modelInfo.ImageName, model)
					if err != nil {
						log.WithError(err).Error("Put db error")
						continue
					}
					log.WithField("name", modelInfo.ImageName).Info("The image updated")
				} else {
					err := db.PutModel(modelInfo.ImageName, modelInfo)
					if err != nil {
						log.WithError(err).Error("Put db error")
						continue
					}
					log.WithField("name", modelInfo.ImageName).Info("The image add")
				}
				//if !imageMap[modelInfo.ImageName] && modelInfo.PublishStatus == models.ModelPublishStatusYes {
				//	log.WithField("model image name", modelInfo.ImageName).Info("pulling image")
				//	go m.dockerOp.PullImage(model.ImageName)
				//}
			}
			m.IsInit = true
			ticker = time.NewTicker(time.Minute * 10)
		}
	}
}

func (m *ModelHandler) GetRpcModelsResp() (*nodemanagerV2.ModelsInfo, error) {
	installedModels := make([]*nodemanagerV2.InstalledModel, 0)
	runningModels := make([]*nodemanagerV2.RunningModel, 0)
	readModels, err := db.GetAllModels()
	if err != nil {
		log.WithError(err).Error("Error reading models")
		return nil, err
	}
	for _, model := range readModels {
		if model.IsInstalled {
			diskSize, err := strconv.ParseInt(model.HardwareRequire.DiskSize, 10, 64)
			if err != nil {
				return nil, err
			}
			model := &nodemanagerV2.InstalledModel{
				ModelId:       strconv.FormatUint(model.TaskId, 10),
				DiskSize:      diskSize,
				InstalledTime: model.SetupTime,
				LastRunTime:   model.LastRunTime,
			}
			installedModels = append(installedModels, model)
		}
		if model.IsRunning {
			model := &nodemanagerV2.RunningModel{
				ModelId: strconv.FormatUint(model.TaskId, 10),
			}
			runningModels = append(runningModels, model)
		}
	}
	res := &nodemanagerV2.ModelsInfo{
		InstalledModels: installedModels,
		RunningModels:   runningModels,
	}
	return res, nil
}

func (m *ModelHandler) MonitorModelStatus() {
	ticker := time.NewTicker(time.Millisecond)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			{
				if !m.IsInit {
					continue
				}
				log.Info("Start monitoring model status for update image info")
				imageList, _ := m.dockerOp.PsImages()
				if imageList != nil && len(imageList) > 0 {
					for _, image := range imageList {
						keys := image.RepoTags
						for _, key := range keys {
							model, _ := db.GetModel(key)
							if model != nil && !model.IsInstalled {
								model.ImageId = image.ID
								model.SetupTime = time.Now().Unix()
								model.IsInstalled = true
								err := db.PutModel(key, model)
								if err != nil {
									continue
								}
								log.WithField("name", model.ImageName).Info("Monitor model status update image info")
							}
						}
					}
				}
				log.Info("Start monitoring model status for update container info")
				containerList := m.dockerOp.ListContainer()
				if containerList != nil && len(containerList) > 0 {
					for _, container := range containerList {
						key := container.Image
						model, err := db.GetModel(key)
						if err != nil || model == nil {
							continue
						}
						if container.State == "running" && !model.IsRunning {
							model.ContainerId = container.ID
							model.LastRunTime = time.Now().Unix()
							model.IsRunning = true
							err = db.PutModel(key, model)
							if err != nil {
								continue
							}
							log.WithField("name", model.ImageName).Info("Monitor model status update container info")
						}
					}
				}
				ticker = time.NewTicker(time.Minute * 10)
			}
		}
	}
}

func (m *ModelHandler) ScanModelsResp() (*nodemanagerV2.ModelsInfo, error) {
	installedModels := make([]*nodemanagerV2.InstalledModel, 0)
	runningModels := make([]*nodemanagerV2.RunningModel, 0)
	//images, err := m.dockerOp.PsImageNameMap()
	//if err != nil {
	//	log.WithError(err).Error("get images failed")
	//	return nil, err
	//}
	//containerList := m.dockerOp.ListContainer()
	//if containerList == nil || len(containerList) == 0 {
	//	log.Error("Get container failed")
	//	return nil, fmt.Errorf("get containe failed")
	//}
	//allModels, err := db.GetAllModels()
	//if err != nil {
	//	log.WithError(err).Error("Get all models failed")
	//	return nil, fmt.Errorf("get all models failed")
	//}
	//for _, model := range allModels {
	//	isExist := images[model.ImageName]
	//	if !isExist {
	//		continue
	//	}
	//	diskSize, err := strconv.ParseInt(model.HardwareRequire.DiskSize, 10, 64)
	//	if err != nil {
	//		continue
	//	}
	//	installedModels = append(installedModels, &nodemanagerV2.InstalledModel{ModelId: strconv.FormatUint(model.TaskId, 10), DiskSize: diskSize, InstalledTime: model.SetupTime, LastRunTime: model.LastRunTime})
	//
	//	containerIsExist := false
	//	for _, container := range containerList {
	//		if model.ImageName == container.Image {
	//			containerIsExist = true
	//		}
	//	}
	//	if containerIsExist {
	//		runningModels = append(runningModels, &nodemanagerV2.RunningModel{ModelId: strconv.FormatUint(model.TaskId, 10), GpuSeq: model.GpuSeq, GpuRam: model.RunningMem, StartedTime: model.LastRunTime, LastWorkTime: model.LastWorkTime, TotalRunCount: model.TotalRunCount, ExecTime: model.EstimatExeTime})
	//	}
	//}
	res := &nodemanagerV2.ModelsInfo{
		InstalledModels: installedModels,
		RunningModels:   runningModels,
	}
	return res, nil
}
