package nm

import (
	"example.com/m/conf"
	"example.com/m/db"
	"example.com/m/largeModel"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"fmt"
	nodemanagerV2 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"strconv"
	"time"
)

var ModelRunningBeforeMem map[string]int64

func init() {
	ModelRunningBeforeMem = make(map[string]int64, 0)
}

type NodeManagerHandler struct {
	nodeManager   *models.NodeManagerClient
	worker        nodemanagerV2.NodeManagerService_RegisterWorkerClient
	msgRespWorker *RespMsgWorker
	taskMsgWorker *TaskWorker
}

func NewNodeManagerHandler(nodeManager *models.NodeManagerClient, worker nodemanagerV2.NodeManagerService_RegisterWorkerClient, msgRespWorker *RespMsgWorker, taskMsgWorker *TaskWorker) *NodeManagerHandler {
	return &NodeManagerHandler{
		nodeManager:   nodeManager,
		worker:        worker,
		msgRespWorker: msgRespWorker,
		taskMsgWorker: taskMsgWorker,
	}
}

func (n *NodeManagerHandler) DistributionMsgWorker(nodeManagerMsgChan chan *nodemanagerV2.ManagerMessage, modelsHandler *largeModel.ModelHandler) {
	for {
		select {
		case rev := <-nodeManagerMsgChan:
			{
				if !n.nodeManager.Status {
					log.Warn("handlerMsg -> node manager is not running")
					return
				}

				heartbeatReq := rev.GetHeartbeatRequest()
				if heartbeatReq != nil {
					unix := time.Unix(int64(heartbeatReq.Timestamp), 0)
					since := time.Since(unix)
					RunningState.NmDelayTime = since.Milliseconds()
					n.nodeManager.UpdateLastHeartTime(time.Now())
					params := utils.BuildParams(heartbeatReq.Timestamp)
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, HeartbeatResp, params)
					log.Info("-------------Heart beat req:-------------", heartbeatReq)
					continue
				}

				taskMsg := rev.GetPushTask()
				if taskMsg != nil {
					go func(msgRespWorker *RespMsgWorker, taskMsgWorker *TaskWorker, taskMsg *nodemanagerV2.PushTaskMessage) {
						isCanExecute, bootUpTime, queueWaitTime, executeTime, imageName := taskMsgWorker.GetAckResp(taskMsg)
						ackParams := utils.BuildParams(taskMsg.TaskId, isCanExecute, bootUpTime, queueWaitTime, executeTime)
						msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, RespTaskAck, ackParams)
						if !isCanExecute {
							log.WithField("taskId", taskMsg.TaskId).Error("The task is not executed")
							return
						}
						taskMsgWorker.Wg.Add(1)
						taskMsgWorker.TaskMsg <- taskMsg
						taskMsgWorker.Wg.Wait()
						taskExecResInterface, _ := taskMsgWorker.LruCache.Get(taskMsg.TaskId)
						//log.WithField("result", taskExecResInterface).Info("lru cache get task result")
						taskExecRes := &models.TaskResult{
							TaskHttpStatusCode: 200,
							TaskRespBody:       nil,
							TaskHttpHeaders:    nil,
							TaskIsSuccess:      false,
							TaskExecTime:       0,
							TaskExecError:      "",
						}
						if taskExecResInterface != nil {
							taskExecRes = taskExecResInterface.(*models.TaskResult)
						}
						isSuccess := taskExecRes.TaskIsSuccess
						containerSign := make([]byte, 0)
						if taskExecRes.TaskRespBody != nil {
							containerSign = taskMsgWorker.DockerOp.GetContainerSign(taskMsg, taskExecRes.TaskRespBody)
							if containerSign == nil || len(containerSign) == 0 {
								log.Error("Container signing failed................")
								isSuccess = false
								taskExecRes.TaskExecError = fmt.Sprintf("%s-%s", "Container sign failed", taskExecRes.TaskExecError)
							}
						} else {
							isSuccess = false
							taskExecRes.TaskExecError = fmt.Sprintf("worker:%s-%s-%s", conf.GetConfig().SignPublicAddress.Hex(), "Task exec error", taskExecRes.TaskExecError)
						}
						reqHash, respHash, minerSign := taskMsgWorker.GetMinerSign(taskMsg, taskExecRes.TaskRespBody)
						taskResultParams := utils.BuildParams(taskMsg.TaskId, containerSign, minerSign, taskExecRes, isSuccess)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.TaskType, taskMsg.TaskType)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ContainerSign, containerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.MinerSign, minerSign)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.ReqHash, reqHash)
						taskMsgWorker.LruCache.Add(taskMsg.TaskId+models.RespHash, respHash)
						msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, SubmitResultResp, taskResultParams)
						RunningState.CompletedTaskCount++
						log.Info("Completed task count: ", RunningState.CompletedTaskCount)
						log.Info("--------------taskMsg--------------:", taskMsg)
						model, _ := db.GetModel(imageName)
						if model != nil {
							params := utils.BuildParams(model)
							n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, RunningModelStatusResp, params)
						}
						msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, GpuUsageResp, ackParams)
					}(n.msgRespWorker, n.taskMsgWorker, taskMsg)
					continue
				}

				deviceUsageMsg := rev.GetDeviceUsageRequest()
				if deviceUsageMsg != nil {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DeviceUsageResp, nil)
					log.Info(deviceUsageMsg)
					continue
				}

				nodeInfoMsg := rev.GetNodeInfoRequest()
				if nodeInfoMsg != nil {
					nodeInfoParam := utils.BuildParams(modelsHandler)
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, NodeInfoResp, nodeInfoParam)
					log.Info(nodeInfoMsg)
					continue
				}

				gpuUsageMsg := rev.GetGpuUsageRequest()
				if gpuUsageMsg != nil {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, GpuUsageResp, nil)
					log.Info(gpuUsageMsg)
					continue
				}

				modelListMsg := rev.GetModelListRequest()
				if modelListMsg != nil {
					modelListParam := utils.BuildParams(modelsHandler)
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, ModelListResp, modelListParam)
					log.Info(modelListMsg)
					continue
				}

				modelOpMsg := rev.GetModelOperateRequest()
				if modelOpMsg != nil {
					go func(modelOpMsg *nodemanagerV2.ModelOperateRequest, dockerOp *operate.DockerOp) {
						operates := modelOpMsg.GetModelOperates()
						for _, modelOp := range operates {
							go n.MonitorImageOp(modelOp)
							model, err := db.GetModel(modelOp.ImageName)
							if err != nil {
								log.WithError(err).Error("Op model - get model error")
								return
							}
							switch modelOp.Operate {
							case nodemanagerV2.ModelOperateType_INSTALL:
								{
									//go dockerOp.PullImage(modelOp.ImageName)
								}
							case nodemanagerV2.ModelOperateType_DELETE:
								{
									//if model.ContainerId != "" {
									//	isRunning := dockerOp.ContainerIsRunning(model.ContainerId)
									//	if isRunning {
									//		dockerOp.StopAndDeleteContainer(model.ContainerId)
									//	}
									//}
									//go dockerOp.RmImage(modelOp.ImageName)
								}
							case nodemanagerV2.ModelOperateType_RUN:
								{
									//dockerCmd := &models.DockerCmd{
									//	HostIp:   models.ZeroHost,
									//	HostPort: n.taskMsgWorker.getExternalPort(),
									//}
									//containerId, gpuSeq, err := dockerOp.CreateAndStartContainer(model, dockerCmd)
									//if err != nil {
									//	log.WithError(err).Error("Error creating container")
									//	continue
									//}
									model.ContainerId = "1111111"
									model.GpuSeq = 0
								}
							case nodemanagerV2.ModelOperateType_STOP:
								{
									//if model.ContainerId != "" {
									model.ContainerId = ""
									//	dockerOp.StopContainer(model.ContainerId)
									//}
								}
							}
							err = db.PutModel(model.ImageName, model)
							if err != nil {
								log.WithError(err).Error("Db put model failed")
							}
						}
					}(modelOpMsg, n.taskMsgWorker.DockerOp)
					continue
				}

				deviceInfoMsg := rev.GetDeviceInfoRequest()
				if deviceInfoMsg != nil {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DeviceInfoResp, nil)
					log.Info(modelListMsg)
					continue
				}

				goodByeMsg := rev.GetGoodbyeMessage()
				if goodByeMsg != nil {
					reason := goodByeMsg.GetReason()
					log.Infof("Server endpoint:%s , good bye reason : %s", n.nodeManager.Endpoint, reason)
					n.nodeManager.UpdateStatus(false)
					log.Warn("Update nm status is false")
					continue
				}
			}
		}
	}
}

func (n *NodeManagerHandler) MonitorImageOp(op *nodemanagerV2.ModelOperate) {
	model, err := db.GetModel(op.ImageName)
	if err != nil {
		log.WithError(err).Error("Op model - get model error")
		return
	}
	ticker := time.NewTicker(time.Second * 2)
	defer ticker.Stop()
	//isOp := false
	switch op.Operate {
	case nodemanagerV2.ModelOperateType_INSTALL:
		{
			//now := time.Now()
			for {
				select {
				case <-ticker.C:
					//if time.Since(now).Hours() > models.OneHour || isOp {
					//	return
					//}
					//imagesMap, err := n.taskMsgWorker.DockerOp.PsImageNameMap()
					//if err != nil {
					//	log.WithError(err).Error("Ps image name map failed")
					//	return
					//}
					//if imagesMap[op.ImageName] {
					//	isOp = true
					time.Sleep(time.Minute * 1)
					model.IsInstalled = true
					model.SetupTime = time.Now().Unix()
					diskSize, _ := strconv.ParseInt(model.HardwareRequire.DiskSize, 10, 64)
					params := utils.BuildParams(&nodemanagerV2.InstalledModel{ModelId: strconv.FormatUint(model.TaskId, 10), DiskSize: diskSize, InstalledTime: model.SetupTime, LastRunTime: model.LastRunTime})
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, AddModelInstalledResp, params)
					break
					//}
				}
			}
		}
	case nodemanagerV2.ModelOperateType_DELETE:
		{
			//now := time.Now()
			for {
				select {
				case <-ticker.C:
					//if time.Since(now).Minutes() > models.OneMinutes || isOp {
					//	return
					//}
					//imagesMap, err := n.taskMsgWorker.DockerOp.PsImageNameMap()
					//if err != nil {
					//	log.WithError(err).Error("Ps image name map failed")
					//	return
					//}
					//if !imagesMap[op.ImageName] {
					//isOp = true
					time.Sleep(time.Second * 10)
					model.IsInstalled = false
					model.IsRunning = false
					model.ContainerId = ""
					params := utils.BuildParams(strconv.FormatUint(model.TaskId, 10))
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DelModelInstalledResp, params)
					break
					//}
				}
			}
		}
	case nodemanagerV2.ModelOperateType_RUN:
		{
			now := time.Now()
			for {
				select {
				case <-ticker.C:
					//if time.Since(now).Minutes() > models.TwoMinutes || isOp {
					//	return
					//}
					//info := GetHardwareInfo()
					//if info == nil {
					//	continue
					//}
					//memIsChange := false
					//for _, gpuInfo := range info.GPU {
					//	if gpuInfo.Seq == model.GpuSeq {
					//		if ModelRunningBeforeMem[op.ImageName] <= gpuInfo.MemFree {
					//			break
					//		}
					//		model.RunningMem = ModelRunningBeforeMem[op.ImageName] - gpuInfo.MemFree
					//		memIsChange = true
					//	}
					//}
					//if !memIsChange {
					//	continue
					//}
					//listContainers := n.taskMsgWorker.DockerOp.ListContainer()
					//if listContainers != nil && len(listContainers) > 0 {
					//	for _, container := range listContainers {
					//		if container.Image == op.ImageName {
					//isOp = true
					time.Sleep(time.Second * 30)
					model.StartUpTime = int64(time.Since(now).Seconds())
					model.ContainerId = "1111"
					model.IsRunning = true
					model.LastRunTime = time.Now().Unix()
					params := utils.BuildParams(&nodemanagerV2.RunningModel{ModelId: strconv.FormatUint(model.TaskId, 10), GpuSeq: model.GpuSeq, GpuRam: model.RunningMem, StartedTime: model.LastRunTime, LastWorkTime: model.LastWorkTime, TotalRunCount: model.TotalRunCount, ExecTime: model.EstimatExeTime})
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, AddModelRunningResp, params)
					break
					//}
					//}
					//}
				}
			}
		}
	case nodemanagerV2.ModelOperateType_STOP:
		{
			//now := time.Now()
			for {
				select {
				case <-ticker.C:
					//if time.Since(now).Minutes() > models.OneMinutes || isOp {
					//	return
					//}
					//listContainers := n.taskMsgWorker.DockerOp.ListContainer()
					//if listContainers != nil && len(listContainers) > 0 {
					//	isFound := false
					//	for _, container := range listContainers {
					//		if container.Image == op.ImageName {
					//			isFound = true
					//		}
					//	}
					//	if !isFound {
					//isOp = true
					time.Sleep(time.Second * 10)
					model.GpuSeq = 999
					model.IsRunning = false
					model.ContainerId = ""
					params := utils.BuildParams(strconv.FormatUint(model.TaskId, 10))
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, DelModelRunningResp, params)
					params = utils.BuildParams(model.TaskId, model.LastRunTime)
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, InstallModelStatusResp, params)
					break
					//}
					//}
				}
			}
		}
	}
	err = db.PutModel(model.ImageName, model)
	if err != nil {
		log.WithError(err).Error("Db put model failed")
	}
}

func (n *NodeManagerHandler) MonitorStandardTaskWorker() {
	ticker := time.NewTicker(time.Minute * 5)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			{
				if n.taskMsgWorker.IsExecStandardTask {
					continue
				}
				if !n.taskMsgWorker.IsExecAiTask {
					n.msgRespWorker.RegisterMsgResp(n.nodeManager, n.worker, FetchStandardTaskResp, nil)
					break
				}
			}
		}
	}
}
