package nm

import (
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	nodeManagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"io"
	"net/http"
	"strings"
	"time"
)

func monitorModelInfo(dockerOp *operate.DockerOp) {
	client := &http.Client{}
	ticker := time.NewTicker(time.Second * 1)
	for {
		select {
		case <-ticker.C:
			modelResp, err := client.Get(conf.GetConfig().ApiUrl)
			if err != nil {
				log.Error("Error getting model info from client failed:", err)
				continue
			}
			bodyBytes, err := io.ReadAll(modelResp.Body)
			if err != nil {
				log.Error("Error reading model response failed:", err)
				continue
			}
			resp := &models.Resp{}
			err = json.Unmarshal(bodyBytes, resp)
			if err != nil {
				log.Error("Unmarshal model response failed:", err)
				continue
			}
			if resp.Code != http.StatusOK {
				log.Error("Response code :", resp.Code)
				continue
			}
			if resp.Data == nil || len(resp.Data) == 0 {
				log.Warn("Response data is empty")
				continue
			}
			modelInfosResp := resp.Data
			imageNameMap, err := dockerOp.PsImageNameMap()
			if err != nil {
				log.Error("Docker op ps images failed:", err)
				continue
			}
			reportTaskIds := make([]uint64, 0)
			maxLong := uint64(0)
			for _, modelInfo := range modelInfosResp {
				if modelInfo.ImageName == "" {
					continue
				}
				split := strings.Split(modelInfo.ImageName, ":")
				if len(split) != 2 {
					continue
				}
				if !imageNameMap[modelInfo.ImageName] {
					// todo: 判断机器资源是否够用
					isPull := isResourceEnough(modelInfo)
					// todo： 如果够用
					if isPull && modelInfo.PublishStatus == models.ModelPublishStatusYes {
						log.WithField("model image name", modelInfo.ImageName).Info("pulling image")
						go dockerOp.PullImage(modelInfo)
					}
				} else {
					log.WithField("name", modelInfo.ImageName).Info("The image name is already")
					reportTaskIds = append(reportTaskIds, modelInfo.TaskId)
					if modelInfo.TaskId > maxLong {
						maxLong = modelInfo.TaskId
					}
				}
				dockerOp.SignApi[modelInfo.ImageName] = modelInfo.SignUrl
			}
			dockerOp.ModelsInfo = modelInfosResp
			reportTaskIds = append(reportTaskIds, maxLong)
			dockerOp.ReportTaskIds = reportTaskIds
			dockerOp.ModelTaskIdIndexesChan <- reportTaskIds
			ticker = time.NewTicker(time.Minute * 10)
		}
	}
}

func reportModelInfo(nodeManager *models.NodeManagerClient,
	worker nodeManagerV1.NodeManagerService_RegisterWorkerClient,
	msgRespWorker *RespMsgWorker, dockerOp *operate.DockerOp) {
	for {
		select {
		case taskIdIndexes := <-dockerOp.ModelTaskIdIndexesChan:
			if !nodeManager.Status {
				log.WithField("endpoint", nodeManager.Endpoint).Error("Node manager is down , stop report model info")
				return
			}
			params := buildParams(taskIdIndexes)
			msgRespWorker.RegisterMsgResp(nodeManager, worker, SubmitResourceMapRes, params)
		}
	}
}

func isResourceEnough(modelInfo *models.ModelInfo) bool {
	return true
}
