package nm

import (
	"context"
	"example.com/m/conf"
	"example.com/m/largeModel"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	nodemanagerV2 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"google.golang.org/grpc"
	"time"
)

type MonitorNm struct {
	NodeManagerClientChan chan *models.NodeManagerClient
	NodeManagerMsgChan    chan *nodemanagerV2.ManagerMessage
	DockerOp              *operate.DockerOp
	IsInit                bool
	ModelHandler          *largeModel.ModelHandler
}

func NewMonitorNm(dockerOp *operate.DockerOp, modelHandler *largeModel.ModelHandler) *MonitorNm {
	return &MonitorNm{
		NodeManagerClientChan: make(chan *models.NodeManagerClient, 10),
		NodeManagerMsgChan:    make(chan *nodemanagerV2.ManagerMessage, 1000),
		DockerOp:              dockerOp,
		ModelHandler:          modelHandler,
		IsInit:                false,
	}
}

func (m *MonitorNm) monitorNmClient() {
	log.Info("Monitoring worker thread start......")
	for {
		select {
		case managerClient := <-m.NodeManagerClientChan:
			go func(nodeManager *models.NodeManagerClient) {
				worker, err := nodeManager.Client.RegisterWorker(context.Background(), grpc.EmptyCallOption{})
				if err != nil {
					log.Error("Registration worker failed", err)
					nodeManager.UpdateStatus(false)
					log.Warn("Update nm status is false")
					return
				}

				msgRespWorker := NewMsgRespWorker()
				go msgRespWorker.SendMsgWorker()
				log.Info("Send msg worker started.......................")

				taskMsgWorker := NewTaskWorker(m.DockerOp)
				taskMsgWorker.DistributionTaskWorker(4)
				log.Info("Distribution task worker started.......................")

				registerRespParam := utils.BuildParams(m.ModelHandler)
				msgRespWorker.RegisterMsgResp(nodeManager, worker, RegisterInfoResp, registerRespParam)
				log.Info("------------------------Send register message ended------------------------")

				msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceInfoResp, nil)
				log.Info("------------------------Send deviceInfo message ended------------------------")

				log.Info("------------------------Send once-off message ended------------------------")

				nodeManagerHandler := NewNodeManagerHandler(nodeManager, worker, msgRespWorker, taskMsgWorker)
				log.Info("Report model info started")

				if nodeManager.IsSelected {
					go m.monitorGpuUsage(msgRespWorker, nodeManager, worker)
				}

				go nodeManagerHandler.MonitorStandardTaskWorker()
				log.Info("Monitor standard task worker started")

				// 处理消息
				for i := 0; i < 2; i++ {
					go nodeManagerHandler.DistributionMsgWorker(m.NodeManagerMsgChan, m.ModelHandler)
				}

				log.Info("------------------------Start rev msg worker thread------------------------")
				for {
					if !IsRecvTask {
						log.Warn("User set recv task status is false")
						msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, nil)
						nodeManager.UpdateStatus(false)
						return
					}
					if IsUpdateBenefitAddr {
						benefitAddrUpdateParam := utils.BuildParams(conf.GetConfig().BenefitAddress)
						msgRespWorker.RegisterMsgResp(nodeManager, worker, BenefitAddrUpdateResp, benefitAddrUpdateParam)
						IsUpdateBenefitAddr = false
					}
					sub := time.Now().Sub(nodeManager.GetLastHeartTime()).Seconds()
					log.WithField("time(uint seconds)", sub).Info("Handler nm msg thread monitor heartbeat time")
					rev, err := worker.Recv()
					if int64(sub) > conf.GetConfig().HeartRespTimeSecond || err != nil {
						log.Error("Rev failed:", err)
						//params := buildParams(fmt.Sprint("Rev failed:", err))
						//msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
						nodeManager.UpdateStatus(false)
						log.Error("Node manager heartbeat is over")
						return
					}
					log.Info("---------------------received message success---------------------")
					m.NodeManagerMsgChan <- rev
					log.Info("---------------------The message input channel success---------------------")
				}
			}(managerClient)
		}
	}
}

func (m *MonitorNm) monitorNodeManagerSeed() {
	ticker := time.NewTicker(time.Second * 1)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			seed := conf.GetConfig().NmSeed
			log.Info("Nm seed url:", seed)
			seedServiceClient := operate.ConnNmGrpc(seed)
			if seedServiceClient == nil {
				log.Warn("Connect nm seed service client is nil")
				continue
			}
			list, err := seedServiceClient.ManagerList(context.Background(), &nodemanagerV2.ManagerListRequest{}, grpc.EmptyCallOption{})
			if err != nil {
				log.WithError(err).Warn("Get manager list failed through nm seed service")
				continue
			}
			if list.GetManagers() == nil || len(list.GetManagers()) == 0 {
				log.Warn("Get managers is empty through Nm seed service")
				continue
			}
			for _, node := range list.GetManagers() {
				if isExistNodeManager(node) {
					log.Warn("Node manager is already exist and updated")
					continue
				}
				nodeManagerArr = append(nodeManagerArr, &NodeManager{Info: node, IsUsed: false, IsExist: true})
			}
			m.IsInit = true
			ticker = time.NewTicker(time.Minute * 10)
		}
	}
}

func (m *MonitorNm) monitorGpuUsage(msgRespWorker *RespMsgWorker, nodeManager *models.NodeManagerClient, worker nodemanagerV2.NodeManagerService_RegisterWorkerClient) {
	tick := time.NewTicker(time.Millisecond)
	defer tick.Stop()
	for {
		select {
		case <-tick.C:
			{
				msgRespWorker.RegisterMsgResp(nodeManager, worker, GpuUsageResp, nil)
				tick = time.NewTicker(time.Minute * 10)
			}
		}
	}
}
