package operate

import (
	"bytes"
	"context"
	"encoding/json"
	"example.com/m/conf"
	"example.com/m/db"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/nm"
	"fmt"
	"github.com/docker/docker/api/types"
	"github.com/docker/docker/api/types/container"
	"github.com/docker/docker/client"
	"github.com/docker/go-connections/nat"
	"github.com/ethereum/go-ethereum/common"
	nodemanagerV2 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v2"
	"io"
	"net/http"
	"os/exec"
	"strconv"
	"strings"
	"time"
)

var httpClient *http.Client

type DockerOp struct {
	IsHealthy        bool
	Reason           string
	dockerClient     *client.Client
	UsedExternalPort map[int64]bool
	SignApi          map[string]string
}

func init() {
	httpClient = &http.Client{}
}

func NewDockerOp() *DockerOp {
	dockerClient, err := GetDockerClient()
	if err != nil {
		return &DockerOp{
			IsHealthy: false,
			Reason:    fmt.Sprintf("The connect docker client failed reason:%s", err.Error()),
		}
	}
	return &DockerOp{
		IsHealthy:        true,
		Reason:           "",
		dockerClient:     dockerClient,
		SignApi:          make(map[string]string, 0),
		UsedExternalPort: make(map[int64]bool, 0),
		//RunningImages:          make(map[string]bool, 0),
	}
}

func (d *DockerOp) GetContainerSign(taskMsg *nodemanagerV2.PushTaskMessage, taskRes []byte) []byte {
	reqBody := &models.TaskReq{
		TaskId:     taskMsg.TaskId,
		TaskParam:  taskMsg.TaskParam,
		TaskResult: taskRes,
	}
	body, err := json.Marshal(reqBody)
	if err != nil {
		log.Error("Unable to marshal task info: ", err.Error())
		return nil
	}
	taskCmd := &models.TaskCmd{}
	err = json.Unmarshal(bytes.NewBufferString(taskMsg.TaskCmd).Bytes(), taskCmd)
	if err != nil {
		log.Errorf("failed to unmarshal task cmd: %s", err.Error())
		return nil
	}
	taskCmd.ImageName = fmt.Sprintf("%s-%s", taskCmd.ImageName, conf.GetConfig().OpSys)
	log.WithField("imageName", taskCmd.ImageName).WithField("url", d.SignApi[taskCmd.ImageName]).Info("container sign")
	request, err := http.NewRequest("POST", d.SignApi[taskCmd.ImageName], bytes.NewReader(body))
	if err != nil {
		log.Error("New http request failed: ", err)
		return nil
	}
	resp, err := httpClient.Do(request)
	if err != nil {
		log.Error("HTTP request failed: ", err)
		return nil
	}
	all, err := io.ReadAll(resp.Body)
	if err != nil {
		log.Error("Failed to read docker response body:", err)
		return nil
	}
	res := &models.ComputeResult{}
	err = json.Unmarshal(all, res)
	if err != nil {
		log.Error("Failed to parse docker response body")
		return nil
	}
	sign := res.Data
	log.WithField("code", res.Code).WithField("msg", res.Msg).WithField("Data", res.Data).Info("Container sign:")
	return common.Hex2Bytes(sign)
}

func (d *DockerOp) ContainerIsRunning(containerId string) bool {
	inspectContainer := d.inspectContainer(containerId)
	if inspectContainer == nil {
		return false
	}
	return inspectContainer.State.Running
}

func (d *DockerOp) ListContainer() []types.Container {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	containers, err := d.dockerClient.ContainerList(ctx, types.ContainerListOptions{})
	if err != nil {
		log.Error("Get container list failed:", err)
		return nil
	}
	for _, c := range containers {
		t := time.Unix(c.Created, 0)
		formattedTime := t.Format("2006-01-02 15:04:05")
		log.Infof("ID: %s, Image: %s, Status: %s , CreateTime: %s \n", c.ID[:10], c.Image, c.Status, formattedTime)
	}
	return containers
}

func (d *DockerOp) CreateAndStartContainer(info *nodemanagerV2.HardwareInfo, modelInfo *models.ModelInfo, dockerCmd *models.DockerCmd) (string, int32, error) {
	gpuSeq := d.checkGpuUsage(info, modelInfo, dockerCmd)
	containerId, err := d.CreateContainer(modelInfo.ImageName, dockerCmd)
	if err != nil {
		log.Error("Error creating container image failed: ", err)
		return "", gpuSeq, err
	}

	// 启动容器
	startContainerIsSuccess := d.StartContainer(containerId)
	if !startContainerIsSuccess {
		log.Error("start container failed:", startContainerIsSuccess)
		return "", gpuSeq, fmt.Errorf("start container failed")
	}

	//ctx, cancel := context.WithTimeout(context.Background(), time.Second*60)
	//defer cancel()
	//statusCh, errCh := d.dockerClient.ContainerWait(ctx, containerId, container.WaitConditionNotRunning)
	//select {
	//case err := <-errCh:
	//	if err != nil {
	//		panic(err)
	//	}
	//case <-statusCh:
	//	break
	//}
	//
	//out, err := d.dockerClient.ContainerLogs(ctx, containerId, types.ContainerLogsOptions{ShowStdout: true})
	//if err != nil {
	//	panic(err)
	//}
	//
	//_, err = stdcopy.StdCopy(os.Stdout, os.Stderr, out)
	//if err != nil {
	//	log.Error("std out put failed:", err)
	//	return "", err
	//}

	return containerId, gpuSeq, nil
}

func (d *DockerOp) CreateContainer(imageName string, dockerCmd *models.DockerCmd) (string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	log.Info("creating container host info:", dockerCmd)
	portBinds := []nat.PortBinding{
		{
			HostIP:   dockerCmd.HostIp,
			HostPort: dockerCmd.HostPort,
		},
	}
	exposePortSet := make(nat.PortSet)
	natPort, _ := nat.NewPort("tcp", strconv.FormatInt(dockerCmd.ContainerPort, 10))
	exposePortSet[natPort] = struct{}{}
	portMap := make(nat.PortMap)
	portMap[natPort] = portBinds
	env := make([]string, 0)
	if dockerCmd.EnvMap != nil && len(dockerCmd.EnvMap) > 0 {
		for key, val := range dockerCmd.EnvMap {
			env = append(env, fmt.Sprintf("%s=%s", key, val))
		}
	}
	resp, err := d.dockerClient.ContainerCreate(ctx, &container.Config{
		ExposedPorts: exposePortSet,
		Image:        imageName,
		Tty:          false,
		Env:          env,
	}, &container.HostConfig{
		PortBindings: portMap,
		AutoRemove:   true, // 容器停止后自动删除
		Resources: container.Resources{
			DeviceRequests: []container.DeviceRequest{
				{
					Driver:       "nvidia",
					Count:        -1, // -1 means all GPUs
					Capabilities: [][]string{{"gpu"}},
					Options:      nil,
				},
			},
		},
	}, nil, nil, "")

	if err != nil {
		log.Error("Error creating container image failed: ", err)
		return "", err
	}
	containerId := resp.ID
	log.Info("Container created with ID:", containerId)

	return containerId, nil
}

func (d *DockerOp) StartContainer(containerID string) bool {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	info, err := d.getContainerInfo(containerID)
	if err == nil {
		for _, port := range info.Ports {
			d.UsedExternalPort[int64(port.PublicPort)] = true
		}
	}
	mounts := info.Mounts
	for _, mount := range mounts {
		if mount.Destination == "/path/to/gpu/memory" {
		}
	}
	// 启动容器
	if err := d.dockerClient.ContainerStart(ctx, containerID, types.ContainerStartOptions{}); err != nil {
		log.Error("Start container failed:", err)
		return false
	}
	log.Info("Container started successfully.")

	return true
}

func (d *DockerOp) StopContainer(containerID string) bool {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	info, err := d.getContainerInfo(containerID)
	if err == nil {
		for _, port := range info.Ports {
			d.UsedExternalPort[int64(port.PublicPort)] = false
		}
	}
	// 停止容器（如果正在运行）
	if err := d.dockerClient.ContainerStop(ctx, containerID, container.StopOptions{}); err != nil {
		// 可能容器已经停止或不存在
		log.Info("Error stopping container:", err)
		return false
	}
	log.Info("Container stopped successfully.")
	return true
}

func (d *DockerOp) StopAndDeleteContainer(containerID string) bool {
	// 停止容器（如果正在运行）
	stopContainer := d.StopContainer(containerID)
	if stopContainer {
		d.RmContainer(containerID)
	}
	log.Info("Container stopped successfully.")
	return true
}

func (d *DockerOp) RmContainer(containerID string) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	// 删除容器
	if err := d.dockerClient.ContainerRemove(ctx, containerID, types.ContainerRemoveOptions{}); err != nil {
		panic(err)
	}
	log.Info("Container deleted successfully.")
}

func (d *DockerOp) PsImages() ([]types.ImageSummary, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	images, err := d.dockerClient.ImageList(ctx, types.ImageListOptions{})
	if err != nil {
		return nil, err
	}
	for _, c := range images {
		log.Infof("%s %s\n", c.ID[:10], c.ParentID)
	}
	return images, nil
}

func (d *DockerOp) PsImageNameMap() (map[string]bool, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	images, err := d.dockerClient.ImageList(ctx, types.ImageListOptions{})
	if err != nil {
		return nil, err
	}
	res := make(map[string]bool, 0)
	for _, image := range images {
		for _, tag := range image.RepoTags {
			res[tag] = true
		}
	}
	return res, nil
}

func (d *DockerOp) PullImage(imageName string) {
	response, err := d.dockerClient.ImagePull(context.Background(), imageName, types.ImagePullOptions{})
	if err != nil {
		log.Errorf("Error pulling image from %s: %v", imageName, err)
		return
	}
	defer func(response io.ReadCloser) {
		err := response.Close()
		if err != nil {
			log.WithError(err).Error("Close image pull response failed")
		}
	}(response)
	log.Info("Image pulled successfully.")
}

func (d *DockerOp) RmImage(imageName string) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	imageId := ""
	images, _ := d.PsImages()
	for _, image := range images {
		for _, tag := range image.RepoTags {
			if tag == imageName {
				imageId = image.ID
				break
			}
		}
	}
	// 删除镜像
	d.dockerClient.ImageRemove(ctx, imageId, types.ImageRemoveOptions{})
	log.Info("Image deleted successfully.")
}

func (d *DockerOp) inspectContainer(containerId string) *types.ContainerJSON {
	ctx, cancel := context.WithTimeout(context.Background(), time.Minute*20)
	defer cancel()
	// 容器信息
	containerJson, err := d.dockerClient.ContainerInspect(ctx, containerId)
	if err != nil {
		log.Error("Container inspect failed: ", err)
		return nil
	}
	log.Info("Image deleted successfully.")
	return &containerJson
}

func (d *DockerOp) GetDockerInfo() (int64, int64, int64, int64, error) {
	cmd := exec.Command("df", "/")
	out, err := cmd.Output()
	if err != nil {
		return 0, 0, 0, 0, fmt.Errorf("exec cmd 'df /' error:%v", err)
	}
	// 将输出按换行符分割成多行
	lines := strings.Split(string(out), "\n")
	// 提取第二行，即包含文件系统信息的行
	if len(lines) >= 2 {
		fields := strings.Fields(lines[1]) // 将行按空格分割成多个字段
		if len(fields) >= 6 {
			log.WithField("Filesystem:", fields[0]).WithField("Size:", fields[1]).WithField("Used:", fields[2]).WithField("Avail:", fields[3]).WithField("Use%:", fields[4]).WithField("Mounted on:", fields[5]).Info()
			totalSize, err := strconv.ParseInt(fields[1], 10, 64)
			usedSize, err := strconv.ParseInt(fields[2], 10, 64)
			availSize, err := strconv.ParseInt(fields[3], 10, 64)
			usageSize, err := strconv.ParseInt(fields[4], 10, 64)
			if err != nil {
				return 0, 0, 0, 0, fmt.Errorf("failed to parse disk size error:%v", err)
			}
			return totalSize, usedSize, availSize, usageSize, nil
		}
	}
	return 0, 0, 0, 0, fmt.Errorf("get disk size failed")
}

func (d *DockerOp) getContainerInfo(id string) (types.Container, error) {
	listContainer := d.ListContainer()
	for _, containerInfo := range listContainer {
		if containerInfo.ID == id {
			return containerInfo, nil
		}
	}
	return types.Container{}, fmt.Errorf("get container info failed")
}

func (d *DockerOp) checkGpuUsage(info *nodemanagerV2.HardwareInfo, modelInfo *models.ModelInfo, dockerCmd *models.DockerCmd) int32 {
	envMap := make(map[string]string, 0)
	gpu := info.GPU
	isMatch := false
	for _, gpuInfo := range gpu {
		if gpuInfo.MemFree > modelInfo.RunningMem {
			envMap[models.CudaEnv] = strconv.FormatInt(int64(gpuInfo.Seq), 10)
			isMatch = true
			break
		}
	}
	if !isMatch {
		runningModel := db.GetRunningModel()
		if len(runningModel) == 0 {
			return 0
		}
		for _, modelInfo := range runningModel {
			if modelInfo.RunningMem > modelInfo.RunningMem {
				isMatch = true
				d.StopContainer(modelInfo.ContainerId)
				envMap[models.CudaEnv] = strconv.FormatInt(int64(modelInfo.GpuSeq), 10)
				break
			}
		}
	}
	if isMatch {
		nm.ModelRunningBeforeMem[modelInfo.ImageName] = dockerCmd.RunningBeforeMem
		gpuSeq, _ := strconv.ParseInt(dockerCmd.EnvMap[models.CudaEnv], 10, 32)
		return int32(gpuSeq)
	}
	return 0
}
