package nm

import (
	"context"
	"example.com/m/conf"
	"example.com/m/log"
	"example.com/m/models"
	"example.com/m/operate"
	"example.com/m/utils"
	"example.com/m/validator"
	nodeManagerV1 "github.com/odysseus/odysseus-protocol/gen/proto/go/nodemanager/v1"
	"google.golang.org/grpc"
	"time"
)

type MonitorNm struct {
	NodeManagerClientChan chan *models.NodeManagerClient
	NodeManagerMsgChan    chan *nodeManagerV1.ManagerMessage
	DockerOp              *operate.DockerOp
	IsInit                bool
}

func NewMonitorNm(DockerOp *operate.DockerOp) *MonitorNm {
	return &MonitorNm{
		NodeManagerClientChan: make(chan *models.NodeManagerClient, 10),
		NodeManagerMsgChan:    make(chan *nodeManagerV1.ManagerMessage, 1000),
		DockerOp:              DockerOp,
		IsInit:                false,
	}
}

func (m *MonitorNm) monitorNmClient() {
	log.Info("Monitoring worker thread start......")
	for {
		select {
		case managerClient := <-m.NodeManagerClientChan:
			go func(nodeManager *models.NodeManagerClient) {
				worker, err := nodeManager.Client.RegisterWorker(context.Background(), grpc.EmptyCallOption{})
				if err != nil {
					log.Error("Registration worker failed", err)
					nodeManager.UpdateStatus(false)
					log.Warn("Update nm status is false")
					return
				}

				msgRespWorker := NewMsgRespWorker()
				go msgRespWorker.SendMsgWorker()
				log.Info("Send msg worker started.......................")

				taskMsgWorker := NewTaskWorker(m.DockerOp)
				taskMsgWorker.DistributionTaskWorker(4)
				log.Info("Distribution task worker started.......................")

				msgRespWorker.RegisterMsgResp(nodeManager, worker, RegisterInfoResp, nil)
				log.Info("------------------------Send register message ended------------------------")

				msgRespWorker.RegisterMsgResp(nodeManager, worker, DeviceInfoResp, nil)
				log.Info("------------------------Send deviceInfo message ended------------------------")

				if len(m.DockerOp.ReportModelIds) == 0 {
					params := utils.BuildParams(m.DockerOp.ReportModelIds, []uint64{0})
					msgRespWorker.RegisterMsgResp(nodeManager, worker, SubmitResourceMapRes, params)
				}
				log.Info("------------------------Send once-off message ended------------------------")

				nodeManagerHandler := NewNodeManagerHandler(nodeManager, worker, msgRespWorker, taskMsgWorker)
				log.Info("Report model info started")

				go nodeManagerHandler.ReportResourceMap(m.DockerOp)
				log.Info("Monitor resource map worker started")

				go nodeManagerHandler.MonitorStandardTaskWorker()
				log.Info("Monitor standard task worker started")

				proofWorker := validator.NewProofWorker()

				// 证明存储
				//go proofWorker.ProofStorage()
				//log.Info("Proof storage worker started")

				// 证明提交
				//go proofWorker.CommitWitness()
				//log.Info("Proof commit worker started")

				// 处理消息
				for i := 0; i < 2; i++ {
					go nodeManagerHandler.DistributionMsgWorker(m.NodeManagerMsgChan, proofWorker)
				}

				log.Info("------------------------Start rev msg worker thread------------------------")
				for {
					sub := time.Now().Sub(nodeManager.GetLastHeartTime()).Seconds()
					log.WithField("time(uint seconds)", sub).Info("Handler nm msg thread monitor heartbeat time")
					rev, err := worker.Recv()
					if int64(sub) > conf.GetConfig().HeartRespTimeSecond || err != nil {
						log.Error("Rev failed:", err)
						//params := buildParams(fmt.Sprint("Rev failed:", err))
						//msgRespWorker.RegisterMsgResp(nodeManager, worker, GoodbyeResp, params)
						nodeManager.UpdateStatus(false)
						log.Error("Node manager heartbeat is over")
						return
					}
					log.Info("---------------------received message success---------------------")
					m.NodeManagerMsgChan <- rev
					log.Info("---------------------The message input channel success---------------------")
				}
			}(managerClient)
		}
	}
}

func (m *MonitorNm) monitorNodeManagerSeed() {
	ticker := time.NewTicker(time.Second * 1)
	for {
		select {
		case <-ticker.C:
			seed := conf.GetConfig().NmSeed
			log.Info("Nm seed url:", seed)
			seedServiceClient := operate.ConnNmGrpc(seed)
			if seedServiceClient == nil {
				log.Warn("Connect nm seed service client is nil")
				continue
			}
			list, err := seedServiceClient.ManagerList(context.Background(), &nodeManagerV1.ManagerListRequest{}, grpc.EmptyCallOption{})
			if err != nil {
				log.WithError(err).Warn("Get manager list failed through nm seed service")
				continue
			}
			if list.GetManagers() == nil || len(list.GetManagers()) == 0 {
				log.Warn("Get managers is empty through Nm seed service")
				continue
			}
			for _, node := range list.GetManagers() {
				if isExistNodeManager(node) {
					log.Warn("Node manager is already exist and updated")
					continue
				}
				nodeManagerArr = append(nodeManagerArr, &NodeManager{Info: node, IsUsed: false, IsExist: true})
			}
			m.IsInit = true
			ticker.Reset(time.Minute * 10)
		}
	}
}
